#######################################################################
# Copyright (C) 2009 VMWare, Inc.
# All Rights Reserved
########################################################################
#
# Bulletin.py
#

__all__ = ['Bulletin', 'BulletinCollection']

import datetime
import logging
import os
import shutil
import time

from . import Errors
from .Utils import XmlUtils
from . import Vib
from . import VibCollection

etree  = XmlUtils.FindElementTree()

#TODO: use schema for validation?
#SCHEMADIR = os.path.join(os.path.dirname(__file__), "schemas")

class ContentBody(object):
   """Represents the <contentBody> tag in a notification bulletin.
         Attributes:
            * html - HTML text content.
            * text - Plain text content.
   """

   def __init__(self, html="", text=""):
      self.html = html
      self.text = text

   def __cmp__(self, other):
      compare = lambda x, y: (x > y) - (x < y)
      return compare((self.html, self.text), (other.html, other.text))

   __lt__ = lambda self, other: self.__cmp__(other) < 0
   __le__ = lambda self, other: self.__cmp__(other) <= 0
   __eq__ = lambda self, other: self.__cmp__(other) == 0
   __ne__ = lambda self, other: self.__cmp__(other) != 0
   __ge__ = lambda self, other: self.__cmp__(other) >= 0
   __gt__ = lambda self, other: self.__cmp__(other) > 0

   @classmethod
   def FromXml(cls, xml):
      if etree.iselement(xml):
         node = xml
      else:
         try:
            node = etree.fromstring(xml)
         except Exception as e:
            msg = "Error parsing XML data: %s." % e
            raise Errors.BulletinFormatError(msg)

      # Note: This implies that the HTML tags are XML-encoded. I.e., we do not
      #       just handle the element content as CDATA.
      html = (node.findtext("htmlData") or "").strip()
      text = (node.findtext("defaultText") or "").strip()

      return cls(html, text)

   def ToXml(self):
      root = etree.Element("contentBody")
      if self.html:
         etree.SubElement(root, "htmlData").text = self.html
      if self.text:
         etree.SubElement(root, "defaultText").text = self.text
      return root


class RecallResolution(object):
   """Represents the <recallResolution> tag in a notification bulletin.
         Attributes:
            * recallfixid - The recall ID.
            * bulletins   - A set containing IDs of fixed bulletins.
   """

   def __init__(self, recallfixid, bulletins=None):
      self.recallfixid = recallfixid
      self.bulletins = bulletins is not None and bulletins or set()

   def __cmp__(self, other):
      compare = lambda x, y: (x > y) - (x < y)

      return compare((self.recallfixid, self.bulletins),
                     (other.recallfixid, other.bulletins))

   __lt__ = lambda self, other: self.__cmp__(other) < 0
   __le__ = lambda self, other: self.__cmp__(other) <= 0
   __eq__ = lambda self, other: self.__cmp__(other) == 0
   __ne__ = lambda self, other: self.__cmp__(other) != 0
   __ge__ = lambda self, other: self.__cmp__(other) >= 0
   __gt__ = lambda self, other: self.__cmp__(other) > 0

   @classmethod
   def FromXml(cls, xml):
      if etree.iselement(xml):
         node = xml
      else:
         try:
            node = etree.fromstring(xml)
         except Exception as e:
            msg = "Error parsing XML data: %s." % e
            raise Errors.BulletinFormatError(msg)

      recallfixid = (node.findtext("recallFixID") or "").strip()
      if not recallfixid:
         raise Errors.BulletinFormatError("Invalid recall fix ID.")
      bulletins = set()
      for bulletinid in node.findall("bulletinIDList/bulletinID"):
         newid = (bulletinid.text or '').strip()
         if newid:
            bulletins.add(newid)

      return cls(recallfixid, bulletins)

   def ToXml(self):
      root = etree.Element("recallResolution")
      etree.SubElement(root, "recallFixID").text = self.recallfixid
      if self.bulletins:
         node = etree.SubElement(root, "bulletinIDList")
         for bulletinid in self.bulletins:
            etree.SubElement(node, "bulletinID").text = bulletinid
      return root


class RecallResolutionList(object):
   """Represents the <resolvedRecalls> tag in a notification bulletin.
         Attributes:
            * recallid    - The ID of the recall.
            * resolutions - A list of RecallResolution objects.
   """
   def __init__(self, recallid, resolutions = None):
      self.recallid = recallid
      self.resolutions = resolutions is not None and resolutions or list()

   def __cmp__(self, other):
      compare = lambda x, y: (x > y) - (x < y)
      return compare((self.recallid, self.resolutions),
                     (other.recallid, other.resolutions))

   __lt__ = lambda self, other: self.__cmp__(other) < 0
   __le__ = lambda self, other: self.__cmp__(other) <= 0
   __eq__ = lambda self, other: self.__cmp__(other) == 0
   __ne__ = lambda self, other: self.__cmp__(other) != 0
   __ge__ = lambda self, other: self.__cmp__(other) >= 0
   __gt__ = lambda self, other: self.__cmp__(other) > 0

   @classmethod
   def FromXml(cls, xml):
      if etree.iselement(xml):
         node = xml
      else:
         try:
            node = etree.fromstring(xml)
         except Exception as e:
            msg = "Error parsing XML data: %s." % e
            raise Errors.BulletinFormatError(msg)

      recallid = (node.findtext("recallID") or "").strip()
      if not recallid:
         raise Errors.BulletinFormatError("Invalid recall ID.")

      resolutions = [RecallResolution.FromXml(x) for x in
                     node.findall("recallResolution")]

      return cls(recallid, resolutions)

   def ToXml(self):
      root = etree.Element("resolvedRecalls")
      etree.SubElement(root, "recallID").text = self.recallid
      for resolution in self.resolutions:
         root.append(resolution.ToXml())
      return root

class Bulletin(object):
   """A Bulletin defines a set of Vib packages for a patch, update,
      or ESX extension.

      Class Variables:
         * NOTIFICATION_RECALL
         * NOTIFICATION_RECALLFIX
         * NOTIFICATION_INFO
         * NOTIFICATION_TYPES

         * RELEASE_PATCH
         * RELEASE_ROLLUP
         * RELEASE_UPDATE
         * RELEASE_EXTENSION
         * RELEASE_NOTIFICATION
         * RELEASE_UPGRADE

         * SEVERITY_CRITICAL
         * SEVERITY_SECURITY
         * SEVERITY_GENERAL

       Attributes:
           * id             - A string specifying the unique bulletin ID.
           * vendor         - A string specifying the vendor/publisher.
           * summary        - The abbreviated (single-line) bulletin summary
                              text.
           * severity       - A string specifying the bulletin's severity.
           * urgency        - A string specifying the bulletin's "urgency."
           * category       - A string specifying the bulletin's category.
           * releasetype    - A string specifying the release type.
           * componentname  - A string specifying the component name.
           * description    - The (multi-line) bulletin description text.
           * kburl          - A URL to a knowledgebase article related to the
                              bulletin.
           * contact        - Contact information for the bulletin's publisher.
           * releasedate    - An integer or float value giving the bulletin's
                              release date/time. May be None if release date is
                              unknown.
           * platforms      - A list of tuples, each of which holds a
                              (version, locale, productLineID) tuple.
           * vibids         - A set of VIB IDs with no corresponding VIB
                              object. These would generally be from a <vibList>
                              XML element, and are meant to be referenced later
                              in order to properly establish keys in the
                              Bulletin with proper VIB objects as values.
   """
   NOTIFICATION_RECALL = 'recall'
   NOTIFICATION_RECALLFIX = 'recallfix'
   NOTIFICATION_INFO = 'info'
   NOTIFICATION_TYPES = (NOTIFICATION_RECALL, NOTIFICATION_RECALLFIX,
                         NOTIFICATION_INFO)

   RELEASE_PATCH = 'patch'
   RELEASE_ROLLUP = 'rollup'
   RELEASE_UPDATE = 'update'
   RELEASE_EXTENSION = 'extension'
   RELEASE_NOTIFICATION = 'notification'
   RELEASE_UPGRADE = 'upgrade'
   RELEASE_TYPES = (RELEASE_PATCH, RELEASE_ROLLUP, RELEASE_UPDATE,
         RELEASE_EXTENSION, RELEASE_NOTIFICATION, RELEASE_UPGRADE)

   SEVERITY_CRITICAL = 'critical'
   SEVERITY_SECURITY = 'security'
   SEVERITY_GENERAL = 'general'
   SEVERITY_TYPES = (SEVERITY_GENERAL, SEVERITY_SECURITY, SEVERITY_CRITICAL)

   URGENCY_CRITICAL = 'critical'
   URGENCY_IMPORTANT = 'important'
   URGENCY_MODERATE = 'moderate'
   URGENCY_LOW = 'low'
   URGENCY_TYPES = (URGENCY_CRITICAL, URGENCY_IMPORTANT, URGENCY_MODERATE,
         URGENCY_LOW)

   def __init__(self, id, **kwargs):
      """Class constructor.
            Parameters:
               * id     - A string giving the unique ID of the Bulletin.
               * kwargs - A list of keyword arguments used to populate the
                          object's attributes.
            Returns: A new Bulletin instance.
      """
      assert id is not None, "id parameter cannot be None"
      self._id = id
      tz = XmlUtils.UtcInfo()
      now = datetime.datetime.now(tz=tz)

      self.vendor         = kwargs.pop('vendor', '')
      self.summary        = kwargs.pop('summary', '')
      self.severity       = kwargs.pop('severity', '')
      self.urgency        = kwargs.pop('urgency', '')
      self.category       = kwargs.pop('category', '')
      self.releasetype    = kwargs.pop('releasetype', '')
      self.componentname  = kwargs.pop('componentname', '')
      self.description    = kwargs.pop('description', '')
      self.kburl          = kwargs.pop('kburl', '')
      self.contact        = kwargs.pop('contact', '')
      self.releasedate    = kwargs.pop('releasedate', now)
      self.platforms      = kwargs.pop('platforms', list())
      self.vibids         = kwargs.pop('vibids', set())

      # These are specific to notification bulletins.
      self.recalledvibs = kwargs.pop('recalledvibs', set())
      self.recalledbulletins = kwargs.pop('recalledbulletins', set())
      self.contentbody = kwargs.pop('contentbody', None)
      self.resolvedrecalls = kwargs.pop('resolvedrecalls', list())

      if kwargs:
         badkws = ', '.join("'%s'" % kw for kw in kwargs)
         raise TypeError("Unrecognized keyword argument(s): %s." % badkws)

   __repr__ = lambda self: self.__str__
   __hash__ = lambda self: hash(self._id)

   id = property(lambda self: self._id)

   # needed for id in set(Bulletin) test
   def __eq__(self, other):
      return self._id == str(other)

   def __str__(self):
      return etree.tostring(self.ToXml()).decode()

   def __add__(self, other):
      """Merge this bulletin with another to form a new object consisting
         of the attributes and VIB list from the newer bulletin.

            Parameters:
               * other - another Bulletin instance.
            Returns: A new Bulletin instance.
            Raises:
               * RuntimeError        - If attempting to add bulletins with
                                       different IDs, or attempting to add an
                                       object that is not a Bulletin object.
               * BulletinFormatError - If attempting to add two bulletins with
                                       the same ID and release date, but which
                                       are not identical.
      """
      if not isinstance(other, self.__class__):
         msg = "Operation not supported for type %s." % other.__class__.__name__
         raise RuntimeError(msg)

      if self.id != other.id:
         raise RuntimeError("Cannot merge bulletins with different IDs.")

      metaattrs = ('vendor', 'summary', 'severity', 'urgency','category',
            'releasetype', 'componentname', 'description', 'kburl', 'contact',
            'releasedate', 'platforms', 'vibids', 'recalledvibs',
            'recalledbulletins', 'contentbody', 'resolvedrecalls')

      if self.releasedate > other.releasedate:
         newer = self
      elif self.releasedate < other.releasedate:
         newer = other
      else:
         for attr in metaattrs:
            if getattr(self, attr) != getattr(other, attr):
               msg = ("Duplicate definitions of bulletin %s with unequal "
                      "attributes." % self.id)
               raise Errors.BulletinFormatError(msg)

         if self.vibids != other.vibids:
            msg = ("Duplicate definitions of bulletin %s with unequal VIB "
                   "lists." % self.id)
            raise Errors.BulletinFormatError(msg)
         # Doesn't really matter which one is 'newer' at this point...
         newer = self

      ret = Bulletin(self.id)
      for attr in metaattrs:
         setattr(ret, attr, getattr(newer, attr))

      return ret

   @classmethod
   def _XmlToKwargs(cls, xml):
      kwargs = {}
      for tag in  ('kbUrl', 'kburl'):
         tagval=(xml.findtext(tag) or "").strip()
         if tagval is not "":
            kwargs[tag.lower()] = tagval
            break

      for tag in  ('id', 'vendor', 'summary', 'severity', 'urgency',
                   'componentName', 'description', 'contact'):
         kwargs[tag.lower()] = (xml.findtext(tag) or "").strip()

      # VUM's spec uses CamelCase for valid enumeration values in the schema,
      # but then uses lowercase in the examples. So just convert to lowercase.
      # https://wiki/SYSIMAGE:Patch_recall_notification_bulletin_XML_schema
      for tag in ('category', 'releaseType'):
         kwargs[tag.lower()] = (xml.findtext(tag) or "").strip().lower()

      rd = (xml.findtext("releaseDate") or "").strip()
      if rd:
         try:
            kwargs['releasedate'] = XmlUtils.ParseXsdDateTime(rd)
         except Exception as e:
            bullid = kwargs.pop('id', 'unkown')
            msg = 'Bulletin %s has invalid releaseDate: %s' % (bullid, e)
            raise Errors.BulletinFormatError(msg)
      else:
         #Set release date if it is not in the input
         now = datetime.datetime.now(tz=XmlUtils.UtcInfo())
         kwargs['releasedate'] = now

      kwargs['platforms'] = list()
      for platform in xml.findall('platforms/softwarePlatform'):
         kwargs['platforms'].append((platform.get('version', ''),
                                     platform.get('locale', ''),
                                     platform.get('productLineID', '')))

      kwargs['vibids'] = set()
      for vibid in xml.findall('vibList/vibID') + xml.findall('vibList/vib/vibID'):
         newid = (vibid.text or '').strip()
         if newid:
            kwargs['vibids'].add(newid)

      if kwargs['releasetype'] == cls.RELEASE_NOTIFICATION:
         if kwargs['category'] == cls.NOTIFICATION_RECALL:
            recalledvibs = set()
            for vibid in xml.findall('recalledVibList/vibID'):
               newid = (vibid.text or '').strip()
               if newid:
                  recalledvibs.add(newid)
            if recalledvibs:
               kwargs['recalledvibs'] = recalledvibs

            recalledbulletins = set()
            for bulletinid in xml.findall('recalledBulletinList/bulletinID'):
               newid = (bulletinid.text or '').strip()
               if newid:
                  recalledbulletins.add(newid)
            if recalledbulletins:
               kwargs['recalledbulletins'] = recalledbulletins
         elif kwargs['category'] == cls.NOTIFICATION_RECALLFIX:
            for node in xml.findall('resolvedRecalls'):
               kwargs['resolvedrecalls'] = RecallResolutionList.FromXml(node)

         node = xml.find('contentBody')
         if node is not None:
            kwargs['contentbody'] = ContentBody.FromXml(node)

      return kwargs

   @classmethod
   def FromXml(cls, xml, **kwargs):
      """Creates a Bulletin instance from XML.

            Parameters:
               * xml    - Must be either an instance of ElementTree, or a
                          string of XML-formatted data.
               * kwargs - Initialize constructor arguments from keywords.
                          Primarily useful to provide default or required
                          arguments when XML data is from a template.
            Returns: A new Bulletin object.
            Exceptions:
               * BulletinFormatError - If the given xml is not a valid XML, or
                                       does not contain required elements or
                                       attributes.
      """
      if etree.iselement(xml):
         node = xml
      else:
         try:
            node = etree.fromstring(xml)
         except Exception as e:
            msg = "Error parsing XML data: %s." % e
            raise Errors.BulletinFormatError(msg)

      kwargs.update(cls._XmlToKwargs(node))
      bullid = kwargs.pop('id', '')

      return cls(bullid, **kwargs)

   def ToXml(self, vibs=None):
      """Serializes the object to XML

            Parameters:
               * vibs - A VibCollection object. If given, will write bulletin
                        with detailed VIB info. If no vibs is passed, write
                        bulletin with vibID only for vibList.
            Returns: An ElementTree.Element object
            Exceptions:
               * BulletinBuildError - If the VIB id can not be resolved in the
                                      given vibs VibCollection.
      """
      root = etree.Element('bulletin')
      for tag in ('id', 'vendor', 'summary', 'severity', 'category', 'urgency',
            'releaseType', 'description', 'kbUrl', 'contact'):
         elem = etree.SubElement(root, tag).text = str(getattr(self, tag.lower()))

      if self.componentname:
         elem = etree.SubElement(root, 'componentName')
         elem.text = self.componentname

      etree.SubElement(root, 'releaseDate').text = self.releasedate.isoformat()

      platforms = etree.SubElement(root, 'platforms')
      for ver, locale, product in self.platforms:
         etree.SubElement(platforms, 'softwarePlatform',
                          version=ver,
                          locale=locale,
                          productLineID=product)

      viblist = etree.SubElement(root, 'vibList')
      for vibid in self.vibids:
         if vibs is None:
            etree.SubElement(viblist, 'vibID').text = vibid
         else:
            try:
               vib = vibs[vibid]
            except KeyError as e:
               msg = 'Can not resolve VIB id %s from VibCollection %s' % (vibid,
                     vibs)
               raise Errors.BulletinBuildError(msg)
            viblist.append(vib.toxml(target=vib.XML_DEST_META))

      if self.releasetype == self.RELEASE_NOTIFICATION:
         if self.category == self.NOTIFICATION_RECALL:
            if self.recalledvibs:
               elem = etree.SubElement(root, 'recalledVibList')
               for vibid in self.recalledvibs:
                  etree.SubElement(elem, 'vibID').text = vibid
            if self.recalledbulletins:
               elem = etree.SubElement(root, 'recalledBulletinList')
               for bulletinid in self.recalledbulletins:
                  etree.SubElement(elem, 'bulletinID').text = bulletinid

         if self.contentbody is not None:
            root.append(self.contentbody.ToXml())

         if self.category == self.NOTIFICATION_RECALLFIX:
            root.append(self.resolvedrecalls.ToXml())

      return root

   def CheckSchema(self):
      #TODO: need to investigate whether to use schema to validate...
      for attr in ('id', 'vendor', 'summary', 'category', 'urgency',
            'description', 'kburl', 'contact', 'vibids'):
         if not getattr(self, attr):
            msg = '%s is required in bulletin' % (attr)
            raise Errors.BulletinFormatError(msg)

      if self.severity not in self.SEVERITY_TYPES:
         msg = 'Unrecognized value "%s", severity must be one of %s.' % (
               self.severity, self.SEVERITY_TYPES)
         raise Errors.BulletinFormatError(msg)

      if self.releasetype not in self.RELEASE_TYPES:
         msg = 'Unrecognized value "%s", releaseType must be one of %s.' % (
               self.releasetype, self.RELEASE_TYPES)
         raise Errors.BulletinFormatError(msg)

   def requiresVibs(self):
      """Checks if the bulletin requires vibs associated with it
         "info" and "recall" bulletins does not have associated vibs.
         "recallFix" ones deliver vibs for fixing the recall.

            Parameters: None
            Returns: Boolean
            Exceptions: None
      """
      return not ((self.releasetype == self.RELEASE_NOTIFICATION) and
                  not (self.category == self.NOTIFICATION_RECALLFIX))


class BulletinCollection(dict):
   """This class represents a collection of Bulletin objects and provides
      methods and properties for modifying the collection.
   """
   def __add__(self, other):
      """Merge this collection with another to form a new collection consisting
         of the union of Bulletins from both.
            Parameters:
               * other - another BulletinCollection instance.
            Returns: A new BulletinCollection instance.
      """
      new = BulletinCollection(self)
      new.update(self)
      for b in other.values():
         new.AddBulletin(b)
      return new

   def AddBulletin(self, bulletin):
      """Add a Bulletin instance to the collection.

      Parameters:
         * bulletin - An Bulletin instance.
      """
      bullid = bulletin.id
      if bullid in self and id(bulletin) != id(self[bullid]):
         self[bullid] += bulletin
      else:
         self[bullid] = bulletin

   def AddBulletinFromXml(self, xml):
      """Add a Bulletin instance based on the xml data.

      Parameters:
         * xml - An instance of ElementTree or an XML string
      Exceptions:
         * BulletinFormatError
      """
      b = Bulletin.FromXml(xml)
      self.AddBulletin(b)

   def AddBulletinsFromXml(self, xml):
      """Add multiple bulletins from an XML file.
            Parameters:
               * xml = An instance of ElementTree or an XML string.
            Exceptions:
               * BulletinFormatError
      """
      if etree.iselement(xml):
         node = xml
      else:
         try:
            node = etree.fromstring(xml)
         except Exception as e:
            msg = "Error parsing XML data: %s." % e
            raise Errors.BulletinFormatError(msg)

      for b in node.findall("bulletin"):
         self.AddBulletinFromXml(b)

   def FromDirectory(self, path, ignoreinvalidfiles = False):
      """Populate this BulletinCollection instance from a directory of Bulletin
         xml files. This method may replace existing Bulletins in the collection.

            Parameters:
               * path               - A string specifying a directory name.
               * ignoreinvalidfiles - If True, causes the method to silently
                                      ignore BulletinFormatError exceptions.
                                      Useful if a directory may contain both
                                      Bulletin xml content and other content.
            Returns: None
            Exceptions:
               * BulletinIOError     - The specified directory does not exist or
                                       cannot be read, or one or more files could
                                       not be read.
               * BulletinFormatError - One or more files were not a valid
                                       Bulletin xml.
      """
      if not os.path.exists(path):
         msg = 'BulletinCollection directory %s does not exist.' % (path)
         raise Errors.BulletinIOError(msg)
      elif not os.path.isdir(path):
         msg = 'BulletinCollection path %s is not a directory.' % (path)
         raise Errors.BulletinIOError(msg)

      for root, dirs, files in os.walk(path, topdown=True):
         for name in files:
            filepath = os.path.join(root, name)
            try:
               with open(filepath) as f:
                  c = f.read()
                  self.AddBulletinFromXml(c)
            except Errors.BulletinFormatError as e:
               if not ignoreinvalidfiles:
                  msg = 'Failed to add file %s to BulletinCollection: %s' % (
                        filepath, e)
                  raise Errors.BulletinFormatError(msg)
            except EnvironmentError as e:
               msg = 'Failed to add Bulletin from file %s: %s' % (filepath, e)
               raise Errors.BulletinIOError(msg)

   def ToDirectory(self, path):
      """Write Bulletin XML in the BulletinCollection to a directory. If the
         directory exists, the content of the directory will be clobbered.

            Parameters:
               * path       - A string specifying a directory name.
            Return: None
            Exceptions:
               * BulletinIOError - The specified directory is not a directory or
                                   cannot create an empty directory
      """
      try:
         if os.path.isdir(path):
             shutil.rmtree(path)
         os.makedirs(path)
      except EnvironmentError as e:
         msg = 'Could not create dir %s for BulletinCollection: %s' % (path, e)
         raise Errors.BulletinIOError(msg)

      if not os.path.isdir(path):
         msg = 'Failed to write BulletinCollection, %s is not a directory.'
         raise Errors.BulletinIOError(msg)

      for b in self.values():
         filepath = os.path.join(path, b.id + '.xml')
         try:
            xml = b.ToXml()
            with open(filepath, 'wb') as f:
               f.write(etree.tostring(xml))
         except EnvironmentError as e:
            msg = 'Failed to write Bulletin xml to %s: %s' % (filepath, e)
            raise Errors.BulletinIOError(msg)
