#!/usr/bin/python
########################################################################
# Copyright (C) 2010,2018 VMWare, Inc.                                 #
# All Rights Reserved                                                  #
########################################################################

import gzip
import logging
import os
import shutil
import sys
import tarfile
import time

if sys.version_info[0] >= 3:
   from io import BytesIO
else:
   try:
      from StringIO import StringIO as BytesIO
   except ImportError:
      # This module sometimes used in an ESXi 4.0.0 environment, which does
      # not have StringIO but it does have cStringIO
      from cStringIO import StringIO as BytesIO

from .Utils.Misc import byteToStr, isString
from . import Errors
from . import ImageProfile
from .Utils import XmlUtils
from . import VibCollection
from . import Vib

etree = XmlUtils.FindElementTree()

log = logging.getLogger(os.path.basename(__file__))

class Database(object):
   """Implements a database to track an Image Profile and its VIBs.

      Class Variables:
         * VISOR_PATH_LIMIT  - The file path length limit on VisorFS.
      Attributes:
         * dbpath    - File path to the Database
         * vibs      - An instance of VibCollection class, representing the VIBs
                       in the database
         * profiles  - An instance of ImageProfileCollection, representing the
                       host ImageProfile
   """
   VISOR_PATH_LIMIT = 99

   def __init__(self, dbpath, dbcreate = True, validate = False,
         addprofile=True):
      """Class constructor.
            Parameters:
               * dbpath   - The location of the esximage database directory.
               * dbcreate - If True, the database will be created if it does
                            not exist. Note that if database files exist in
                            the directory given by dbpath, but cannot be
                            properly read or parsed, an exception will be
                            raised regardless of this parameter's value.
               * addprofile - If True, will create an empty profile if database
                              is created.
            Returns: A new Database instance.
            Raises:
               * DatabaseIOError - If dbpath cannot be created; if dbpath does
                                   not exist and dbcreate is False; if dbpath
                                   exists but is not a directory; or if the
                                   database cannot be read from or written to
                                   the file system.
      """
      self._dbpath = dbpath
      self._vibsdir = os.path.join(dbpath, 'vibs')
      self._profilesdir = os.path.join(dbpath, 'profiles')
      self._validate = validate
      createprofile = False
      if not os.path.exists(dbpath):
         if not dbcreate:
            msg = "Database directory '%s' does not exist." % dbpath
            raise Errors.DatabaseIOError(dbpath, msg)
         else:
            try:
               os.makedirs(self._vibsdir)
               os.makedirs(self._profilesdir)
               createprofile = True
            except Exception as e:
               msg = 'Failed to create empty Database directory: %s' % (e)
               raise Errors.DatabaseIOError(self._dbpath, msg)
      elif not os.path.isdir(dbpath):
         msg = "Database path '%s' exists, but it is not a directory." % dbpath
         raise Errors.DatabaseIOError(dbpath, msg)

      self.vibs = VibCollection.VibCollection()
      self.profiles = ImageProfile.ImageProfileCollection()
      if createprofile and addprofile:
         log.debug('Creating empty profile for db: %s' % (self._dbpath))
         self.profiles.AddProfile(ImageProfile.ImageProfile('EmptyProfile',
            'esximage-auto'))
         self.Save()

   @property
   def dbpath(self):
      return self._dbpath

   @property
   def profile(self):
      if len(self.profiles) == 1:
         return list(self.profiles.values())[0]
      else:
         return None
   @property
   def vibIDs(self):
      return self.profile and self.profile.vibIDs or set()

   def Clear(self):
      """Clear the content of the database."""
      self.vibs.clear()
      self.profiles.clear()

   def Load(self):
      """Populates the bulletins and vibs attributes of this object with
         information from the database. It is safe to call this method
         repeatedly to update this object when the database has changed.
            Exceptions:
               * DatabaseFormatError - If vibs.xml or bulletins.zip cannot be
                                       parsed, or format is otherwise invalid.
               * DatabaseIOError     - If a file cannot be read.
      """
      self.Clear()
      self._LoadDB()

   def Save(self):
      """Writes the information currently in the object back to the database.
            Raises:
               * DatabaseIOError     - On a failure to write to a file.
      """
      self._WriteDB()
      self._Commit()

   def _LoadDB(self):
      try:
         self.vibs.FromDirectory(self._vibsdir, validate = self._validate)
      except Errors.VibFormatError as e:
         msg = 'Could not parse Vib xml from database %s: %s' % (self._dbpath,
               e)
         raise Errors.DatabaseFormatError(msg)
      except Errors.VibIOError as e:
         msg = 'Error reading Vib xml from database %s: %s' % (self._dbpath, e)
         raise Errors.DatabaseIOError(self._dbpath, msg)
      except Exception as e:
         msg = 'Error reading Vib xml from database %s: %s' % (self._dbpath, e)
         raise Errors.DatabaseFormatError(msg)

      try:
         self.profiles.FromDirectory(self._profilesdir, validate = self._validate)
      except Errors.ProfileFormatError as e:
         msg = 'Could not parse ImageProfile from database %s: %s' %  (
               self._dbpath, e)
         raise Errors.DatabaseFormatError(msg)
      except Errors.ProfileIOError as e:
         msg = 'Error reading ImageProfile xml from database %s: %s' % (
               self._dbpath, e)
         raise Errors.DatabaseIOError(self._dbpath, msg)
      except Exception as e:
         msg = 'Error reading ImageProfile xml from database %s: %s' % (
               self._dbpath, e)
         raise Errors.DatabaseFormatError(msg)

   def _WriteDB(self):
      self._newvibsdir = self._vibsdir + '.new'
      self._newprofilesdir = self._profilesdir + '.new'
      try:
         self._createemptydir(self._newvibsdir)
         self._createemptydir(self._newprofilesdir)
      except EnvironmentError as e:
         msg = 'Failed to create temporary DB dir: %s'  % (e)
         raise Errors.DatabaseIOError(self._dbpath, msg)

      try:
         self.vibs.ToDirectory(self._newvibsdir)
      except Errors.VibIOError as e:
         msg = 'Failed to save VIB metadata to dir %s: %s' % (self._newvibsdir,
               e)
         raise Errors.DatabaseIOError(self._dbpath, msg)

      try:
         self.profiles.ToDirectory(self._newprofilesdir, toDB = True)
      except Errors.ProfileIOError as e:
         msg = 'Failed to save Profile data to dir %s: %s'  % (
               self._newprofilesdir, e)
         raise Errors.DatabaseIOError(self._dbpath, msg)

   def _Commit(self):
      assert (os.path.isdir(self._newprofilesdir) and
            os.path.isdir(self._newvibsdir))
      try:
         shutil.rmtree(self._vibsdir)
         shutil.rmtree(self._profilesdir)
      except EnvironmentError as e:
         msg = 'Error in purging old directory: %s' % (e)
         raise Errors.DatabaseIOError(self._dbpath, msg)

      try:
         os.rename(self._newvibsdir, self._vibsdir)
         os.rename(self._newprofilesdir, self._profilesdir)
      except Exception as e:
         msg = 'Failed to commit the change to database: %s' % (e)
         raise Errors.DatabaseIOError(self._dbpath, msg)

   @staticmethod
   def _createemptydir(path):
      if os.path.isdir(path):
         shutil.rmtree(path)
      elif os.path.isfile(path):
         os.unlink(path)
      os.makedirs(path)

class TarDatabase(Database):
   """Implements an Image Profile and VIB database within a tar archive."""
   DB_PREFIX = ('var', 'db', 'esximg')

   def __init__(self, dbpath=None, dbcreate=True, validate = False):
      """Class constructor.
            Parameters:
               * dbpath   - An optional file name or file object containing a
                            targz'ed database. If specified, image profile and
                            VIB data will be loaded from this location.
               * dbcreate - If True, the database will be created if it does
                            not exist. Note that if database files exist in
                            the tar file given by dbpath, but cannot be
                            properly read or parsed, an exception will be
                            raised regardless of this parameter's value. This
                            parameter has no effect if dbpath is unspecified or
                            is a file object.
            Returns: A new TarDatabase instance.
            Raises:
               * DatabaseIOError - If dbpath is specified, but a location
                                   either does not exist or could not be
                                   parsed.
      """
      self._dbpath = dbpath
      #NOTE: tardatabase might be constructed in Windows, we need to use Unix-
      #      style paths.
      self._dbroot = '/'.join(self.DB_PREFIX)
      self._vibsdir = '/'.join([self._dbroot, 'vibs'])
      self._profilesdir = '/'.join([self._dbroot, 'profiles'])
      self._validate = validate

      self.vibs = VibCollection.VibCollection()
      self.profiles = ImageProfile.ImageProfileCollection()

      if (isString(dbpath) and not os.path.exists(dbpath)
          and dbcreate):
         log.debug('Creating tar database at: %s' % dbpath)
         self.Save(dbpath)

   def Load(self, dbpath=None):
      """Populates the bulletins and vibs attributes of this object with
         information from the database. It is safe to call this method
         repeatedly to update this object when the database has changed.
            Parameters:
               * dbpath   - An optional file name or file object containing a
                            targz'ed database. If specified, image profile and
                            VIB data will be loaded from this location. This
                            overrides any value specified in the constructor.
            Exceptions:
               * DatabaseFormatError - If VIB or Image Profile XML data cannot
                                       be parsed.
               * DatabaseIOError     - If a file cannot be read.
      """
      self.Clear()
      dbpath = dbpath is not None and dbpath or self._dbpath
      if dbpath is None:
         msg = "Cannot open tar database: No path specified."
         raise Errors.DatabaseIOError(None, msg)

      try:
         if isString(dbpath):
            dbpath = byteToStr(dbpath)
            t = tarfile.open(dbpath, "r:gz")
         else:
            t = tarfile.open(mode="r:gz", fileobj=dbpath)
      except Exception as e:
         msg = "Failed to open tar database: %s." % e
         raise Errors.DatabaseIOError(dbpath, msg)

      try:
         for info in t:
            if not info.isfile():
               continue

            try:
               f = t.extractfile(info)
               content = f.read()
               f.close()
            except Exception as e:
               msg = "Failed to read %s from tardatabase %s: %s" % (
                     info.name, self._dbpath, e)
               raise Errors.DatabaseIOError(dbpath, msg)

            try:
               if info.name.startswith(self._vibsdir):
                  # skip signature and original descriptor files
                  if not info.name.endswith(Vib.EXTENSION_ORIG_DESC) and \
                     not info.name.endswith(Vib.EXTENSION_ORIG_SIG):
                     sigpath = info.name + Vib.EXTENSION_ORIG_SIG
                     try:
                        member = t.getmember(sigpath)
                     except KeyError:
                        member = None
                     if member is not None:
                        if member.size != 0:
                           sigfile = t.extractfile(member)
                           signature = sigfile.read()
                           sigfile.close()
                        else:
                           signature = None
                     else:
                        signature = None
                     origpath = info.name + Vib.EXTENSION_ORIG_DESC
                     try:
                        member = t.getmember(origpath)
                     except KeyError:
                        member = None
                     if member is not None:
                        if member.size != 0:
                           origfile = t.extractfile(member)
                           origdesc = origfile.read()
                           origfile.close()
                        else:
                           origdesc = None
                     else:
                        origdesc = None
                     self.vibs.AddVibFromXml(content, origdesc = origdesc,
                                             signature = signature,
                                             validate = self._validate)
               elif info.name.startswith(self._profilesdir):
                  self.profiles.AddProfileFromXml(content, self._validate)
               else:
                  log.info('Unrecognized file %s in tardatabase.' % (info.name))
            except Exception as e:
               msg = 'Error parsing VIB/ImageProfile from DB %s: %s - %s' % (
                     self._dbpath, e.__class__.__name__, e)
               raise Errors.DatabaseFormatError(dbpath, msg)
      finally:
         t.close()

   def Save(self, dbpath=None, savesig=False):
      """Writes the information currently in the object back to the database.
            Parameters:
               * dbpath  - If specified, must be either a string specifying a
                           file name, or a file object to write the database to.
                           If not specified, the location used in the class
                           constructor will be used.
               * savesig - If set to True, the original descriptor and its
                           signature will be written to the database.
            Raises:
               * DatabaseIOError - On a failure to write to a file.
      """
      dbpath = dbpath is not None and dbpath or self._dbpath
      if dbpath is None:
         msg = "Cannot write tar database: No path specified."
         raise Errors.DatabaseIOError(None, msg)

      try:
         if isString(dbpath):
            dbpath = byteToStr(dbpath)
            t = tarfile.open(dbpath + ".new", "w:gz")
         else:
            t = tarfile.open(mode="w:gz", fileobj=dbpath)
      except Exception as e:
         msg = "Error opening tar database: %s." % e
         raise Errors.DatabaseIOError(dbpath, msg)

      try:
         self._createemptydb(t)
      except Exception as e:
         msg = "Error creating tar database: %s." % e
         raise Errors.DatabaseIOError(dbpath, msg)

      try:
         for v in self.vibs.values():
            xml = v.ToXml()
            c = etree.tostring(xml)
            fn = '/'.join([self._vibsdir, self.vibs.FilenameForVib(v)])
            self._addfile(t, fn, c)
            c = v.GetSignature()
            sigfile = fn + Vib.EXTENSION_ORIG_SIG
            if c is not None and savesig:
               self._addfile(t, sigfile, c)
            c = v.GetOrigDescriptor()
            origdescfile = fn + Vib.EXTENSION_ORIG_DESC
            if c is not None and savesig:
               if type(c) is str and sys.version_info[0] >= 3:
                   c = c.encode()
               self._addfile(t, origdescfile, c)

         for p in self.profiles.values():
            xml = p.ToXml(toDB=True)
            c = etree.tostring(xml)
            fn = '/'.join([self._profilesdir,
                  self.profiles.FilenameForProfile(p)])
            self._addfile(t, fn, c)

         t.close()
      except Exception as e:
         msg = "Error writing tar database: %s."  % e
         raise Errors.DatabaseIOError(dbpath, msg)

      if isString(dbpath):
         try:
            # Atomic rename is apparently too much to ask for from Windows.
            if os.path.exists(dbpath):
               os.unlink(dbpath)
            os.rename(dbpath + ".new", dbpath)
         except Exception as e:
            msg = "Error renaming temporary tar database: %s." % e
            raise Errors.DatabaseIOError(dbpath, msg)

   @classmethod
   def _addfile(cls, tar, path, content):
      #padding leading root
      if len(path) + 1 >= cls.VISOR_PATH_LIMIT:
         msg = ('database file path is too long (%d),  the limit is %d.\n%s' %
               (len(path) + 1, cls.VISOR_PATH_LIMIT, path))
         raise Errors.DatabaseIOError(msg)

      info = tarfile.TarInfo(path)
      info.mtime = time.time()
      info.mode = 0o644
      info.gid = 0
      info.uid = 0
      info.type = tarfile.REGTYPE
      info.size = len(content)
      tar.addfile(info, BytesIO(content))

   @classmethod
   def _adddir(cls, tar, path):
      info = tarfile.TarInfo(path)
      info.mtime = time.time()
      info.mode = 0o755
      info.type = tarfile.DIRTYPE
      tar.addfile(info)

   def _createemptydb(self, t):
      # Add parent directories which are needed to populate VisorFS
      p = ''
      for d in self.DB_PREFIX:
         if p:
            p = '/'.join([p, d])
         else:
            p = d
         self._adddir(t, p)

      self._adddir(t, self._vibsdir)
      self._adddir(t, self._profilesdir)
