########################################################################
# Copyright (C) 2010 VMWare, Inc.
# All Rights Reserved
########################################################################
from __future__ import with_statement # 2.5 only

"""This module defines a class for managing a collection of VIB depots,
   where each depot consists of index, vendor-index, and metadata.zip files.
"""
import logging
import os
import tempfile
import threading
import sys

if sys.version_info[0] >= 3:
   from urllib.request import url2pathname
   from urllib.parse import urlparse
else:
   from urllib import url2pathname
   from urlparse import urlparse

from . import Depot
from . import Errors
from . import Vib
from . import VibCollection
from . import ImageProfile
from . import Downloader
from .Utils import PathUtils
from .Utils.Misc import isString
from . import Scan


log = logging.getLogger('DepotCollection')

class OfflineBundleNotLocal(Exception):
   pass

class ProfileAlreadyExists(Exception):
   pass

class VibAlreadyExists(Exception):
   pass

## Not a real URL, this is used to describe the depot specially created
#  for custom channels created via AddChannel().
#  Note: it needs to use a supported scheme or urlparse() behavior is unpredictable
CUSTOM_DEPOT_URL = 'file://custom/depot/'

class DepotCollection(object):
   """This class holds a collection of depots and their channels and
      metadata files.  It helps with constructing the depot hierarchy,
      managing the channels and metadata files, and
      querying and filtering on VIBs and image profiles within metadatas
      managed by this collection.

      Attributes:
         * depots   - A list of DepotIndex instances
         * channels - A dict of available channels, with the unique
                      channel ID as the key, and each value being an instance
                      of DepotChannel.
         * vibs     - A VibCollection of all Vib packages contained in all
                      the available channels.
         * profiles - An ImageProfileCollection for all profiles contained in all
                      the available channels.
         * vibscandata - An instance of VibScanner. Holds scan results for all the VIBs.
   """
   def __init__(self, depots=None, cachedir=None):
      """Constructs a new DepotCollection.
         Parameters:
            * depots   - An initial list of DepotIndex instances
                         to populate the DepotCollection with.
            * cachedir - Path to a directory for caching depot metadata files
         Returns:
            An instance of DepotCollection.
      """
      self._depots = Depot.DepotTreeNode(children=depots)
      self.cachedir = cachedir
      self._channels = dict()
      self._vibs = VibCollection.VibCollection()
      self._profiles = ImageProfile.ImageProfileCollection()
      self._vibscandata = Scan.VibScanner()
      self._lock = threading.RLock()

      # Create a depot for custom channels. Make sure everything has a URL.
      self.customvendor = Depot.VendorIndex(name="Custom Depot", code="XXX",
                                            indexfile="custom-depot.xml",
                                            baseurl=CUSTOM_DEPOT_URL)
      self.customdepot = Depot.DepotIndex(url=self._TranslateDepotUrl(CUSTOM_DEPOT_URL),
                                          children=[self.customvendor])

   def _safeget(self, attr):
      # Gets a class attribute with locking
      self._lock.acquire()
      try:
         return getattr(self, attr)
      finally:
         self._lock.release()

   depots = property(lambda self: self._safeget('_depots').children)
   channels = property(lambda self: self._safeget('_channels'))
   vibs = property(lambda self: self._safeget('_vibs'))
   profiles = property(lambda self: self._safeget('_profiles'))
   vibscandata = property(lambda self: self._safeget('_vibscandata'))

   @staticmethod
   def _TranslateDepotUrl(durl):
      # Translates offline .zip file paths to a zip:/ URI
      # Adds /index.xml to the end of depot URLs which don't have it
      # NOTE: does not translate local paths to file:/// URI's
      if durl.lower().endswith('.zip'):
         if os.path.splitdrive(durl)[0] or PathUtils.IsDatastorePath(durl):
            # If this is a Windows path starting with a drive letter, we're
            # already done. We specifically check this before trying urlparse,
            # because urlparse will think the drive letter is the URL scheme.
            # Likewise, if this is a [datastore] path, don't mess with it;
            # PathUtils.CreateZipUrl will handle it.
            path = durl
         else:
            scheme, netloc, path = urlparse(durl)[:3]
            if scheme == '':
               # durl was actually just a relative path. Convert it to an
               # absolute one.
               path = os.path.abspath(durl)
            elif scheme == 'file':
               # durl was a file URL. Use url2pathname to convert the URL path
               # to an OS path.
               path = url2pathname(path)
            else:
               # durl was a URL, but not a file URL. We don't support this.
               raise OfflineBundleNotLocal
         durl = PathUtils.CreateZipUrl(path, 'index.xml')
      elif not durl.lower().endswith('.xml'):
         # Note: forward slashes can be mixed in to Windows pathnames, it
         # still works.
         durl = durl.rstrip('/').rstrip('\\') + '/index.xml'

      return durl


   def ConnectDepots(self, depotUrls, timeout=None, ignoreerror=True):
      """Connects to depots and downloads the entire hierarchy of metadata
         files from index.xml down to every metadata.zip.  Parses all metadata
         files including VIB package and image profile metadata.  The
         vibs, profiles, and vibscandata properties will all be updated.
         Parameters:
            * depotUrls - The full remote URLs of the depot index.xml or local
                          offline bundle. If URL ends with '.zip', the URL will
                          be treated as pointing to offline bundle and local
                          file path is required. If the URL does not end
                          in .xml, it will be assumed to be a directory and
                          'index.xml' will be added.
            * timeout   - Network timeout in seconds
            * ignoreerror - If true, exception will be logged and will continue
                            to try to connect to the next depot URL.
         Returns:
            A tuple.  First is a list of DepotIndex instances for the depots
            connected to.  The second element is a list of error instances
            for URLs that could not be connected to or parsed.  The error
            instances would be either MetadataDownloadError or MetadataFormatError.
            Note: an invalid offline .zip file (not in .zip format) yields
            MetadataDownloadError.
      """

      Downloader.Downloader.setEsxupdateFirewallRule('true')
      try:
         return self._connectDepots(depotUrls, timeout, ignoreerror)
      finally:
         Downloader.Downloader.setEsxupdateFirewallRule('false')

   def _connectDepots(self,depotUrls, timeout=None, ignoreerror=True):
      vendorurls = list()
      connected = list()
      errors = list()
      for durl in depotUrls:
         try:
            durl = self._TranslateDepotUrl(durl)
         except OfflineBundleNotLocal:
            msg = ('local file path is required for bundle zip file :'
                   '(%s) seems to be a remote URI' % (durl))
            log.info(msg)
            e = ValueError(msg)
            errors.append(e)
            if not ignoreerror:
               raise e
            continue
         try:
            log.debug("Downloading depot index.xml from %s" % (durl))
            d = Downloader.Downloader(durl, timeout=timeout)
            depotindex = self.ParseDepotIndex(durl, d.Open())
         except Downloader.DownloaderError as e:
            msg = "Could not download from depot at %s, skipping (%s)" % (
                     durl, str(e))
            log.info(msg)
            err = Errors.MetadataDownloadError(durl, '', msg)
            errors.append(err)
            if not ignoreerror:
               raise err
            continue
         except Exception as e:
            log.info("Error parsing depot index.xml from %s: %s" % (
                     durl, str(e)))
            errors.append(e)
            if not ignoreerror:
               raise
            continue

         connected.append(depotindex)
         vendorurls.extend(v.absurl for v in depotindex.children)

      metaurls = list()
      for vurl in vendorurls:
         try:
            log.debug("Downloading vendor index.xml from %s" % (vurl))
            d = Downloader.Downloader(vurl, timeout=timeout)
            vindex = self.ParseVendorIndex(vurl, d.Open())
         except Downloader.DownloaderError as e:
            msg = "Could not download from vendor index at %s, skipping (%s)" % (
                     vurl, str(e))
            log.info(msg)
            err = Errors.MetadataDownloadError(vurl, '', msg)
            errors.append(err)
            if not ignoreerror:
               raise err
            continue
         except Exception as e:
            log.info("Error parsing vendor index.xml from %s: %s" % (
                      vurl, str(e)))
            errors.append(e)
            if not ignoreerror:
               raise
            continue

         metaurls.extend(m.absurl for m in vindex.children)

      for metaurl in metaurls:
         fn = None
         try:
            try:
               log.debug("Downloading metadata.zip from %s" % (metaurl))
               fh, fn = tempfile.mkstemp()
               os.close(fh)
               d = Downloader.Downloader(metaurl, local=fn, timeout=timeout)
               metafile = d.Get()
               metanode = self.ParseMetadata(metaurl, metafile)
            except Downloader.DownloaderError as e:
               msg = "Could not download metadata.zip from %s, skipping (%s)" % (
                        metaurl, str(e))
               log.info(msg)
               err = Errors.MetadataDownloadError(metaurl, '', msg)
               errors.append(err)
               if not ignoreerror:
                  raise err
               continue
            except Exception as e:
               log.info("Error parsing metadata.zip from %s: %s" % (
                  metaurl, str(e)))
               errors.append(e)
               if not ignoreerror:
                  raise
               continue
         finally:
            if fn:
               os.unlink(fn)

      self._buildCollections(scanvibs=True)

      return connected, errors

   def DisconnectDepots(self, depotUrls):
      """Disconnects from one or more depots.  The DepotIndexes specified
         will be removed from the depots property.
         Also, the vibs, profiles, and vibscandata properties will be updated.
         Parameters:
            * depotUrls   : The URLs of the depots to remove.
         Exceptions:
            IndexError - if one of the depotUrls is not found.
      """
      toremove = set(depotUrls)
      delindex = list()
      self._lock.acquire()
      needrebuild = False
      try:
         for i, depot in enumerate(self._depots.children):
            if depot.absurl in toremove:
               # Remove channels
               for vendoridx in depot.children:
                  for chan in vendoridx.channels.values():
                     del self._channels[chan.channelId]
               delindex.append(i)
               toremove.remove(depot.absurl)
               needrebuild = True
         # Delete depots in reverse order
         while delindex:
            del self._depots.children[delindex.pop()]
      finally:
         if needrebuild:
            self._buildCollections(scanvibs=True)
         self._lock.release()
      if len(toremove):
         raise IndexError('URLs %s could not be found in existing depots' %
                          (', '.join(list(toremove))))

   def ParseDepotIndex(self, depotIndexUrl, localpath):
      """Parses the depot index.xml file.  The depot will be added to the
         DepotCollection list or the existing entry will be updated.
         NOTE: Please handle HTTP redirects BEFORE this method is invoked.
               depotIndexUrl should be the redirected URL.  This URL will
               be kept and used for comparison later, so changes in the URL
               may result in duplicate or inconsistent state.
         Parameters:
            * depotIndexUrl - The full remote URL of the depot index.xml file.
                              Used to compute the URLs of vendor-indexes.
            * localpath     - The local path where the index.xml to be parsed
                              resides, or a file object
         Returns:
            The instance of DepotIndex that was created or updated.  To obtain
            the URLs of the vendor indexes, look up the absurl attribute of each
            VendorIndex in the 'children' list attribute of the returned
            instance.
         Exceptions:
            MetadataFormatError - cannot parse metadata
            FileIOError         - cannot access localpath
      """
      try:
         if isString(localpath):
            f = open(localpath, 'r')
         else:
            f = localpath
         indexxml = f.read()
         index = Depot.DepotIndex.FromXml(indexxml, url=depotIndexUrl)
         f.close()
      except EnvironmentError as e:
         raise Errors.FileIOError(localpath, str(e))

      self._lock.acquire()
      try:
         dindex = self._depots.AddChild(index)
      finally:
         self._lock.release()
      return dindex

   def ParseVendorIndex(self, vendorIndexUrl, localpath):
      """Parses a vendor-index.xml file and its channels and metadata file info.
         Inserts the information in the appropriate place in the depot hierarchy.
         Updates the channels attribute with the channels found here.
         NOTE: Please handle HTTP redirects BEFORE this method is invoked.
               depotIndexUrl should be the redirected URL.  This URL will
               be kept and used for comparison later, so changes in the URL
               may result in duplicate or inconsistent state.
         Parameters:
            * vendorIndexUrl - The full remote URL of the vendor-index.xml file.
                               NOTE: This must match the absurl in the child
                               nodes as returned by ParseDepotIndex() above
                               in order for insertion to work correctly.
            * localpath      - The local path where the vendor-index.xml file
                               to be parsed resides, or a file object
         Returns:
            An instance of VendorIndex.  The child nodes contains information
            about metadata URL as well as channels.
         Exceptions:
            MetadataFormatError - cannot parse metadata
            FileIOError         - cannot access localpath
            IndexError - None of the depots contains the vendorIndexUrl passed in.
      """
      index = None
      for depot in self._depots.children:
         #
         # For each depot, does it have the vendor-index at vendorIndexUrl?
         index = depot.GetChildFromUrl(vendorIndexUrl)
         if index:
            try:
               if isString(localpath):
                  f = open(localpath, 'r')
               else:
                  f = localpath
               xml = f.read()
               vidx = Depot.VendorIndex.FromXml(xml, url=vendorIndexUrl)
               f.close()
            except EnvironmentError as e:
               raise Errors.FileIOError(localpath, str(e))

            self._lock.acquire()
            try:
               index = depot.AddChild(vidx)
               self._AddChannels(index)
            finally:
               self._lock.release()
            return index

      # None of the depots contained vendorIndexUrl.  Just give up.
      raise IndexError("Vendor-index URL %s not found in any depot" % (vendorIndexUrl))

   def _AddChannels(self, vendorindex):
      # Prune channels in self._channels that don't belong
      vchannels = vendorindex.channels
      rm_list = []
      for channel in self._channels.values():
         if channel.vendorindex.absurl == vendorindex.absurl:
            if channel.name not in vchannels:
               rm_list.append(channel.channelId)
      for channelId in rm_list:
         del self._channels[channelId]

      # Adds the channels from vendorindex to self._channels
      for channel in vchannels.values():
         self._channels[channel.channelId] = channel

   def ParseMetadata(self, metadataUrl, localpath):
      """Parses a metadata.zip file.  The VIB packages and image profiles
         found in the metadata.zip file will be subsequently available
         for query and scan operations.  The vibs and profiles properties
         will be updated.
         Parameters:
            * metadataUrl - The full remote URL of the metadata.zip file.
                            Will be used to compute VIB remotelocations.
                            NOTE: This must match the absurl in the child
                            nodes as returned by ParseVendorIndex() above
                            in order for insertion to work correctly.
            * localpath   - The local path where the metadata.zip file
                            to be parsed resides.
         Returns:
            An instance of Depot.MetadataNode
         Exceptions:
            MetadataFormatError - cannot parse metadata.zip
            MetadataIOError     - cannot access or read localpath
            IndexError          - None of the vendor-indexes contain the
                                  metadataUrl passed in.
      """
      # Find the right MetadataNode to replace
      # Note this assumes each MetadataNode has a unique absurl
      for depot in self._depots.children:
         for vidx in depot.children:
            metanode = vidx.GetChildFromUrl(metadataUrl)
            if metanode:
               self._lock.acquire()
               try:
                  metanode.ReadMetadataZip(localpath)
                  metanode.SetVibRemoteLocations(metadataUrl)
               finally:
                  self._lock.release()
               self._buildCollections(scanvibs=False)
               return metanode

      raise IndexError("Metadata URL %s not found in the depot" % (metadataUrl))

   def GetChannelsByName(self, names):
      """Returns a set of channel IDs coresponding to the given channel names.
         Parameters:
            * names - an iterable of strings for the channel names to search.
         Returns:
            A set of channel IDs, or the empty set if none of the names
            correspond to an existing channel.  If two channels contain the same
            channel name, they will both be returned.
      """
      channels = set()
      for chanID, channel in self._channels.items():
         if channel.name in names:
            channels.add(chanID)
      return channels

   def GetChannelsByDepotUrl(self, urls):
      """Returns a set of channel IDs for channels in the given depots.
         Parameters:
            * urls - an iterable of depot URLs whose channels should be returned.
         Returns:
            A set of channel IDs, or the empty set of none of the depot URLs
            are valid or contain channels.
      """
      channels = set()
      for url in urls:
         durl = self._TranslateDepotUrl(url)
         if PathUtils.IsAbsolutePath(durl):
            durl = PathUtils.FilepathToUrl(durl)
         depot = self._depots.GetChildFromUrl(durl)
         if depot:
            for vendor in depot.children:
               channels.update(set(c.channelId for c in vendor.channels.values()))
      return channels

   def GetVibsByChannelId(self, channelId):
      """Returns all the VIBs from a given channel as a VibCollection.
         Parameters:
            * channelId - the channel GUID, from the keys of self.channels
                          or channelId property of a DepotChannel
         Returns:
            An instance of VibCollection.
         Raises:
            KeyError, if channelId is not in self.channels
      """
      vibs = VibCollection.VibCollection()
      for meta in self.channels[channelId].metadatas:
         vibs += meta.vibs
      return vibs

   def GetVibsByDepots(self, depotUrls):
      """Returns all the VIBs from one or more depots.
         Parameters:
            * depotUrls - the URLs for the interested depots.
         Returns:
            An instance of VibCollection.
      """
      depotfilter = set(self._TranslateDepotUrl(depotUrl) for depotUrl in depotUrls)
      vibs = VibCollection.VibCollection()
      for depot in self._depots.children:
         if depot.absurl in depotfilter:
            for vendorindex in depot.children:
               for meta in vendorindex.children:
                  vibs += meta.vibs
      return vibs

   def GetProfilesByChannelId(self, channelId):
      """Returns all the image profiles from a given channel.
         Parameters:
            * channelId - the channel GUID, from the keys of self.channels
                          or channelId property of a DepotChannel
         Returns:
            An instance of ImageProfileCollection.
         Raises:
            KeyError, if channelId is not in self.channels
      """
      profiles = ImageProfile.ImageProfileCollection()
      for meta in self.channels[channelId].metadatas:
         profiles += meta.profiles
      return profiles

   def GetProfilesByDepots(self, depotUrls):
      """Returns all the image profiles from one or more depots.
         Parameters:
            * depotUrls - The URLs for the interested depots
         Returns:
            An instance of ImageProfileCollection.
      """
      depotfilter = set(self._TranslateDepotUrl(depotUrl) for depotUrl in depotUrls)
      profiles = ImageProfile.ImageProfileCollection()
      for depot in self._depots.children:
         if depot.absurl in depotfilter:
            for vendorindex in depot.children:
               for meta in vendorindex.children:
                  profiles += meta.profiles
      return profiles

   def ScanVibs(self):
      """Populates the vibscandata property by scanning all the VIB package
         dependencies of the internal vibs collection.
      """
      self._lock.acquire()
      try:
         self._vibscandata = self._vibs.Scan()
      finally:
         self._lock.release()

   def AddChannel(self, name):
      """Adds a new channel to the depotcollection.  This is mostly designed
         for persisting user-created image profiles.  A special URL will be
         created for the channel.
         The channel will be added to a common depot created for the purpose
         of persisting custom channels.  The common depot and all custom
         channels can be removed using DisconnectDepots.
         If the channel already exists, then this is a NOP.
         Parameters:
            * name       - A friendly name for the new channel
         Returns:
            The new DepotChannel instance.
      """
      metaurl = CUSTOM_DEPOT_URL + name
      metanode = Depot.MetadataNode(productId=Depot.DEPOT_PRODUCT,
                                    channels=[name],
                                    url=metaurl)
      log.debug("Adding custom channel %s" % (metanode.absurl))

      with self._lock:
         self._depots.AddChild(self.customdepot)
         self.customvendor.AddChild(metanode)
         self._AddChannels(self.customvendor)
      return self.customvendor.channels[name]

   def AddProfile(self, profile, channelId, replace=True):
      """Adds or replaces an image profile in an existing channel.  The
         channel should have been added using AddChannel above.
         Parameters:
            * profile   - an ImageProfile instance to add/replace.
            * channelId - the channel ID of the channel to affect
            * replace   - if True, allows an existing image profile
                          to be replaced, where existing means an image profile
                          with the same ID.  If False, no profile with the same
                          ID can exist in this DepotCollection.
         Raises:
            KeyError    - if the channelId does not correspond to a valid channel
            ProfileAlreadyExists  - if the image profile already exists and replace is
                             False.
            ValueError  - The image profile to be replaced is read-only and
                          cannot be modified.
      """
      if profile.profileID not in self._profiles:
         # A new profile.  Add it to first meta of the channel.
         with self._lock:
            meta = self._channels[channelId].metadatas[0]
            meta.profiles[profile.profileID] = profile
            self._profiles[profile.profileID] = profile
      elif replace:
         # Profile exists already and we can replace it
         if not self._profiles[profile.profileID].readonly:
            with self._lock:
               for meta in self._channels[channelId].metadatas:
                  if profile.profileID in meta.profiles:
                     meta.profiles[profile.profileID] = profile
               self._profiles[profile.profileID] = profile
         else:
            raise ValueError("Image Profile '%s' is read-only and cannot be updated"
                             % (self._profiles[profile.profileID].name))
      else:
         raise ProfileAlreadyExists("Image Profile '%s' already exists and replace=False"
                             % (self._profiles[profile.profileID].name))

   def RemoveProfile(self, profileID, channelname=''):
      """Removes an image profile from the collection for a channel. If channelname is empty, 
         remove the image profile from all the channels.
         Parameters:
            * profileID - the ID of the profile to remove.
            * channelname - the name of the channel from which to remove the profile.
         Raises:
            KeyError - the profile does not exist in the collection.
            ValueError - the profile is not found in the specified channel
      """
      with self._lock:
         keep = False
         deleted = False
         for chan in self._channels.values():
            for meta in chan.metadatas:
              if profileID in meta.profiles:
                 if len(channelname) == 0 or chan.name == channelname:
                    del meta.profiles[profileID]
                    deleted = True
                 else:
                    keep = True

         # Since we only support removing from customer channel. If the profile is not in
         # the specified channel but in other channels, which means the profile cannot
         # be removed as it is from a real depot.
         if keep and not deleted:
            raise ValueError("The image profile '%s' cannot be removed as it is "
               "from a read-only depot." % (self._profiles[profileID].name))

         # Not in any channels, remove it from top-level
         if not keep:
            del self._profiles[profileID]

   def AddVib(self, vib, channelId, replace=True):
      """Adds or replaces a VIB in an existing channel. The channel should have
         have been added using AddChannel above.
         Parameters:
            * vib       - a BaseVib instance to add/replace.
            * channelId - the channel ID of the channel to affect
            * replace   - if True, allows an existing VIB with the same ID to be
                          replaced. If False, no VIB with the same ID can exist
                          in this DepotCollection.
         Raises:
            KeyError    - if the channelId does not correspond to a valid channel
            VibAlreadyExists  - if the VIB already exists and replace is
                             False.
      """
      if vib.id not in self._vibs:
         # A new VIB. Add it to first meta of the channel.
         with self._lock:
            meta = self._channels[channelId].metadatas[0]
            meta.vibs[vib.id] = vib
            self._vibs[vib.id] = vib
      elif replace:
         # VIB exists already and we can replace it
         with self._lock:
            for meta in self._channels[channelId].metadatas:
               if vib.id in meta.vibs:
                  meta.vibs[vib.id] = vib
            self._vibs[vib.id] = vib
      else:
         raise VibAlreadyExists("VIB '%s' already exists and replace=False"
                                % (self._vibs[vib.id].name))
      self._buildCollections(scanvibs=True)

   def AddVibFromUrl(self, url, channelId, replace=True):
      try:
         d = Downloader.Downloader(url)
         vib = Vib.ArFileVib.FromFile(d.Open())
         vib.remotelocations = [url]
         # We don't know how much time is going to elapse before we try to
         # do anything with VIB payloads (like export an image profile), so
         # just close the file. We'll re-open it when it's needed.
         vib.Close()
      except Downloader.DownloaderError as e:
         raise Errors.VibDownloadError(url, None, str(e))
      self.AddVib(vib, channelId, replace)
      return vib

   def RemoveVib(self, vibID):
      """Removes an VIB from the collection.
         Parameters:
            * vibID - the ID of the VIB to remove.
         Raises:
            KeyError - the VIB does not exist in the collection.
      """
      with self._lock:
         del self._vibs[vibID]
         for chan in self._channels.values():
            for meta in chan.metadatas:
               try:
                  del meta.vibs[vibID]
               except KeyError:
                  pass
      self._buildCollections(scanvibs=True)

   def _buildCollections(self, scanvibs=False):
      # Rebuilds the vibs and profiles collections.
      # If scanvibs is True, also rebuild the vibscandata.

      #
      # It is OK if there are duplicate metadata.zips in different channels.
      # The VibCollection Merge method would simply add the same Vib instances
      # together, which results in a VibCollection with the same contents.
      #
      self._lock.acquire()
      try:
         vibs = VibCollection.VibCollection()
         profiles = ImageProfile.ImageProfileCollection()
         for channel in self._channels:
            vibs += self.GetVibsByChannelId(channel)
            profiles += self.GetProfilesByChannelId(channel)

         self._vibs = vibs
         self._profiles = profiles

         if scanvibs:
            self.ScanVibs()
      finally:
         self._lock.release()
