#!/usr/bin/python
########################################################################
# Copyright (C) 2010-2011, 2015-2016 VMWare, Inc.                      #
# All Rights Reserved                                                  #
########################################################################

import os
import shutil
import tarfile
import tempfile

from vmware.esximage import Bulletin, Database, Errors, Metadata
from vmware.esximage import PayloadTar
from vmware.esximage.Utils import Iso9660, SyslinuxConfig, XmlUtils
from vmware.esximage.Utils.HashedStream import HashedStream, HashError
from vmware.esximage.Utils.Misc import isString, isSeekSupported

from vmware.esximage.ImageBuilder.ImageBuilder import ImageBuilder

PAYLOAD_READ_CHUNKSIZE = 1024 * 1024

etree = XmlUtils.FindElementTree()

"""This module provides a class for creating an ISO9660 image based on an
   image profile."""

class EsxIsoImage(ImageBuilder):
   "This class creates an ISO9660 image with the contents of an image profile."

   def __init__(self, imageprofile):
      """Class constructor.
            Parameters:
               * imageprofile - An instance of ImageProfile. The 'vibs'
                                attribute of the object must contain valid
                                references to the VIBs in the 'vibIDs'
                                property. Those references must include either
                                a file object or a valid remote location.
      """
      ImageBuilder.__init__(self, imageprofile)

   def _AddPayloads(self, volume, checkdigests=True, installer=True):
      # This method adds each payload to the ISO, while doing the right things
      # for various payload types.

      imgpayldfobj = tempfile.TemporaryFile()
      imgpayldtar = PayloadTar.PayloadTar(imgpayldfobj)

      # Must generate unique names for certain payloads:
      self.imageprofile.GenerateVFATNames()
      for vibid in self.imageprofile.vibIDs:
         for payload, fobj in self.imageprofile.vibs[vibid].IterPayloads():
            if payload.payloadtype in (payload.TYPE_VGZ, payload.TYPE_TGZ,
                                       payload.TYPE_BOOT,
                                       payload.TYPE_INSTALLER_VGZ):
               # We assume GenerateVFATNames() is working properly, and that
               # vibstates is populated.
               state = self.imageprofile.vibstates[vibid]
               payloadfn = state.payloads[payload.name]
            elif payload.payloadtype == payload.TYPE_TEXT:
               payloadfn = '-'.join([self.imageprofile.vibs[vibid].vendor,
                                     self.imageprofile.vibs[vibid].name,
                                     payload.name])
            else:
               payloadfn = payload.name

            if not payloadfn:
               msg = "VIB '%s' has payload with no name." % vibid
               raise Errors.VibFormatError(None, msg)

            if checkdigests and not payload.checksums:
               msg = ("Digest checking is enabled, but VIB payload does not "
                      "have a checksum.")
               raise Errors.VibPayloadDigestError(vibid, payload.name, msg)

            # some payloads will be copied twice where seek(0) is used
            # if a payload does not seek, we create a temporary file
            if not isSeekSupported(fobj):
               tfp = tempfile.TemporaryFile()
               bytesread = fobj.read(PAYLOAD_READ_CHUNKSIZE)
               while bytesread:
                  tfp.write(bytesread)
                  bytesread = fobj.read(PAYLOAD_READ_CHUNKSIZE)
               fobj.close()
               tfp.seek(0)
               fobj = tfp

            if checkdigests:
               try:
                  checksum = payload.GetPreferredChecksum()
                  algo = checksum.checksumtype.replace("-", "")
                  fobj = HashedStream(fobj, checksum.checksum, algo)
               except Exception as e:
                  msg = "Unable to calculate digest for VIB payload: %s." % e
                  raise Errors.VibPayloadDigestError(vibid, payload.name, msg)

            if payload.payloadtype in (payload.TYPE_VGZ, payload.TYPE_TGZ,
                                       payload.TYPE_BOOT_COM32_BIOS,
                                       payload.TYPE_TEXT):
               volume.AddFile(fobj, payloadfn, payload.size)
            elif (payload.payloadtype == payload.TYPE_INSTALLER_VGZ
                  and installer):
               volume.AddFile(fobj, payloadfn, payload.size)
            elif payload.payloadtype == payload.TYPE_BOOT:
               imgpayldtar.AddPayload(fobj, payload, payloadfn)
               if checkdigests:
                  fobj.reset()
               else:
                  fobj.seek(0)
               volume.AddFile(fobj, payloadfn, payload.size)
            elif payload.payloadtype == payload.TYPE_UPGRADE and installer:
               volume.AddFile(fobj, "UPGRADE/" + payloadfn, payload.size)
            elif payload.payloadtype == payload.TYPE_BOOT_ISO_BIOS:
               record = volume.AddFile(fobj, payloadfn, payload.size)
               volume.SetBootImage(record)
            elif payload.payloadtype == payload.TYPE_ELTORITO_IMAGE_EFI:
               record = volume.AddFile(fobj, payloadfn, payload.size)
               volume.AddAltBootImage(record)
            elif payload.payloadtype == payload.TYPE_BOOT_LOADER_EFI:
               volume.AddFile(fobj, "EFI/BOOT/" + payloadfn, payload.size)

      imgpayldtar.close()
      imgpayldfobj.seek(0)
      if installer:
         volume.AddFile(imgpayldfobj, self.PAYLOADTAR_NAME)

   def _AddDatabase(self, volume):
      # This method generates a tar database from the image profile, writes it
      # to a temp file, then adds the temp file to the ISO image.
      db = Database.TarDatabase()
      db.profiles.AddProfile(self.imageprofile)
      for vibid in self.imageprofile.vibIDs:
         db.vibs.AddVib(self.imageprofile.vibs[vibid])

      # check if vib signatures must be saved
      savesig = self.imageprofile.IsSecureBootReady()

      try:
         tmpf = tempfile.TemporaryFile()
         db.Save(dbpath=tmpf, savesig=savesig)
         tmpf.seek(0, 0)
      except Errors.EsxupdateError:
         # Preserve stack trace to enable troubleshooting the root cause.
         raise
      except EnvironmentError as e:
         msg = "Could not create temporary database: %s." % e
         raise Errors.DatabaseIOError(None, msg)

      volume.AddFile(tmpf, self.DATABASE_NAME)

   def _AddMetadataZip(self, volume):
      # This method generates a metadata.zip from the image profile.
      metadata = Metadata.Metadata()
      # If we don't create a bulletin, we won't get anything in vmware.xml.
      bulletin = Bulletin.Bulletin(self.imageprofile.name)

      for vibid in self.imageprofile.vibIDs:
         metadata.vibs.AddVib(self.imageprofile.vibs[vibid])
         bulletin.vibids.add(vibid)
      metadata.profiles.AddProfile(self.imageprofile)
      metadata.bulletins.AddBulletin(bulletin)

      try:
         tmpdir = tempfile.mkdtemp()
      except IOError as e:
         msg = "Could not create temporary metadata directory: %s." % e
         raise Errors.MetadataIOError(msg)

      metapath = os.path.join(tmpdir, "metadata.zip")

      # Use nested try/except inside of try/finally to maintain Python 2.4
      # compatibility.
      try:
         try:
            metadata.WriteMetadataZip(metapath)
            # This is not efficient, but it saves us from having to worry about
            # keeping track of the temporary directory and cleaning it up
            # later.
            tmpf = tempfile.TemporaryFile()
            f = open(metapath, "rb")
            shutil.copyfileobj(f, tmpf)
            f.close()
            tmpf.seek(0)
         except IOError as e:
            msg = "Error copying metadata to temporary file: %s." % e
            raise Errors.MetadataIOError(msg)
      finally:
         try:
            shutil.rmtree(tmpdir)
         except:
            pass

      volume.AddFile(tmpf, "upgrade/metadata.zip")

   def _AddProfileXml(self, volume):
      # Adds profile.xml to the upgrade directory on the ISO.
      profilexml = self.imageprofile.ToXml()
      xmltree = etree.ElementTree(profilexml)
      try:
         tmpf = tempfile.TemporaryFile()
         xmltree.write(tmpf)
         tmpf.seek(0)
      except Exception as e:
         # Specifically use Exception here, since we want to catch both I/O
         # errors, as well as XML serialization errors. Because we don't know
         # which specific etree implementation or what underlying parser it
         # will use, we also can't be sure of which exception we will get for
         # XML errors.
         msg = "Error writing temporary profile XML: %s." % e
         raise Errors.ProfileIOError(msg)

      volume.AddFile(tmpf, "upgrade/profile.xml")

   def _AddIsoLinuxConfig(self, volume, installer=True):
      # This method populates isolinux.cfg, using the profile name, and the
      # module order from the image profile. It writes the config to a temp
      # file, then adds the temp file to the ISO image.
      config = SyslinuxConfig.Config()
      config.default = "menu.c32"
      config.menutitle = "%s Boot Menu" % self.imageprofile.name
      config.timeout = 80
      config.nohalt = 1
      config.prompt = False

      label = config.AddLabel("install")
      if installer:
         label.menulabel = "%s ^Installer" % self.imageprofile.name
      else:
         label.menulabel = "%s ^System" % self.imageprofile.name
      label.kernel = "mboot.c32"
      label.append = "-c boot.cfg"

      label = config.AddLabel("hddboot")
      label.menulabel = "^Boot from local disk"
      label.localboot = "0x80"

      tmpf = tempfile.TemporaryFile()
      config.Write(tmpf)
      tmpf.seek(0, 0)

      volume.AddFile(tmpf, "isolinux.cfg")

   def _AddDiscinfo(self, volume):
      # This method populates .discinfo, a file which describes the contents
      # of the iso. The file format is not particualarly well-defined, so we have
      # a simple one with the product name and version
      # The final field, 'Virtual HW:' is used by Easy Install

      tmpf = tempfile.TemporaryFile()
      tmpf.write(("%(product)s\nVersion: %(version)s\n" % \
            {
               "product" : "ESXi",
               "version" : self.imageprofile.GetEsxVersion()
             }).encode())
      tmpf.seek(0, 0)

      volume.AddFile(tmpf, ".discinfo")

   def _AddBootCfg(self, volume, installer=True, features=None):
      # This method populates boot.cfg and efi/boot/boot.cfg.
      #
      # moduleroot is required for efi/boot/boot.cfg, but apparently doesn't hurt
      # anything for /boot.cfg.
      bootcfg = self._GetBootCfg(installer, moduleroot='/', features=features,
                                 isoImage=True)
      if not bootcfg:
         return

      tmpf = tempfile.TemporaryFile()
      bootcfg.write(tmpf)
      tmpf.seek(0, 0)

      volume.AddFile(tmpf, "boot.cfg")

      tmpf = tempfile.TemporaryFile()
      bootcfg.write(tmpf)
      tmpf.seek(0, 0)

      volume.AddFile(tmpf, "efi/boot/boot.cfg")

   def Write(self, f, checkdigests=True, insertmd5=True, installer=True,
             checkacceptance=True, features=None):
      """Write out the ISO 9660 image to a file or file-like object.
            Parameters:
               * f               - A string giving a file name, or an object
                                   implementing the Python file protocol. The
                                   ISO image will be output to this path or
                                   file object.
               * checkdigests    - If True, payload digests will be verified
                                   when the ISO is written. Defaults to True.
               * insertmd5       - If True, an MD5 hash of the ISO contents
                                   will be inserted into the application data
                                   field of the ISO's primary volume
                                   descriptor. This is used by VUM Upgrade
                                   Manager for verifying the integrity of the
                                   image. Note that if this is True, the 'f'
                                   parameter must support rewinding the file
                                   pointer. Defaults to True.
               * installer       - Enable the installer in the booted image.
                                   Defaults to True.
               * checkacceptance - If True, validate the Acceptance Level of
                                   each VIB. If the validation fails, an
                                   exception is raised. Defaults to True.
               * features        - Features state switches to enable or disable.
            Raises:
               * DatabaseIOError       - If unable to write the tar database to
                                         a temporary file.
               * ImageIOError          - If unable to write to a temporary file
                                         or the image output, or unable to
                                         compute the MD5 checksum of the image.
               * ProfileFormatError    - If the image profile has consistency
                                         errors.
               * VibDownloadError      - If unable to download one or more VIBs.
               * VibFormatError        - If one or more VIBs is not in proper
                                         VIB format.
               * VibIOError            - If unable to obtain the location of,
                                         or read data from, a VIB.
               * VibPayloadDigestError - If the calculated digest for one or
                                         more VIB payloads does not match the
                                         value given in VIB metadata.
      """
      if isString(f):
         fobj = open(f, "w+b")
         def removeoutput():
            fobj.close()
            os.unlink(f)
      else:
         fobj = f
         def removeoutput():
            pass

      volume = Iso9660.Iso9660Volume()
      pvd = volume.primaryvolumedescriptor
      if insertmd5:
         # This is where we will store the md5 digest.
         pvd.applicationdata = "\0" * 16
      pvd.volumeid = self.imageprofile.name.upper()
      pvd.applicationid = "ESXIMAGE"

      try:
         self._CheckVibFiles(checkacceptance)
         self._AddPayloads(volume, checkdigests, installer)
         self._AddDatabase(volume)
         self._AddDiscinfo(volume)
         if installer:
            self._AddMetadataZip(volume)
            self._AddProfileXml(volume)
      except:
         removeoutput()
         raise

      try:
         self._AddIsoLinuxConfig(volume, installer)
         self._AddBootCfg(volume, installer, features)
      except IOError as e:
         removeoutput()
         msg = "Error writing boot configuration: %s." % e
         raise Errors.ImageIOError(str(fobj), msg)
      volume.Finalize()

      if insertmd5 and not hasattr(fobj, "seek") or not hasattr(fobj, "tell"):
         removeoutput() # not likely to be an actual file.
         msg = ("Can not insert MD5 digest into ISO image when writing to "
                "non-seekable output.")
         raise Errors.ImageIOError(str(fobj), msg)

      try:
         if insertmd5:
            # Application data is at offset 883 of primary volume descriptor:
            appdataoffset = fobj.tell() + (16 * 2048) + 883
            hashfobj = HashedStream(fobj, method="md5")
            volume.Write(hashfobj)
            isoend = fobj.tell()
            fobj.seek(appdataoffset)
            fobj.write(hashfobj.digest)
            fobj.seek(isoend)
         else:
            volume.Write(fobj)
      except (IOError, HashError, Iso9660.Iso9660Error) as e:
         removeoutput()
         msg = "Error occurred writing ISO image: %s." % e
         raise Errors.ImageIOError(str(fobj), msg)

      if isString(f):
         fobj.close()
