#!/usr/bin/python
########################################################################
# Copyright (C) 2010,2015,2017 VMWare, Inc.                            #
# All Rights Reserved                                                  #
########################################################################

import os
import shutil
import tarfile
import tempfile

from vmware.esximage import Database, Errors, Vib
from vmware.esximage import PayloadTar
from vmware.esximage.Utils.HashedStream import HashedStream

from vmware.esximage.ImageBuilder.ImageBuilder import ImageBuilder

"""This module provides a class for creating a PXE image based on an
   image profile."""

class EsxPxeImage(ImageBuilder):
   "This class creates a PXE image with the contents of an image profile."

   DATABASE_NAME = "imgdb.tgz"
   PAYLOADTAR_NAME = "imgpayld.tgz"

   def __init__(self, imageprofile):
      """Class constructor.
            Parameters:
               * imageprofile - An instance of ImageProfile. The 'vibs'
                                attribute of the object must contain valid
                                references to the VIBs in the 'vibIDs'
                                property. Those references must include either
                                a file object or a valid remote location.
      """
      ImageBuilder.__init__(self, imageprofile)

   @staticmethod
   def _CopyFileObjToFileName(srcfobj, destfpath, length=None):
      destdir = os.path.dirname(destfpath)
      if not os.path.exists(destdir):
         os.makedirs(destdir)

      newfobj = open(destfpath, 'w')
      if length:
         shutil.copyfileobj(srcfobj, newfobj, length)
      else:
         shutil.copyfileobj(srcfobj, newfobj)
      newfobj.close()

   def _DeployVib(self, pxedir, checkdigests=True, installer=True):
      """Deploy all VIBs to the PXE directory.

      Boot-VIBs are extracted to the PXEDIR directory. Other extra VIBs are
      copied to PXEDIR/vibs/.
      """
      imgpayldfobj = tempfile.TemporaryFile()
      imgpayldtar = PayloadTar.PayloadTar(imgpayldfobj)

      # Must generate unique names for certain payloads:
      self.imageprofile.GenerateVFATNames()
      for vibid in self.imageprofile.vibIDs:
         if not self.imageprofile.vibstates[vibid].boot:
            vib = self.imageprofile.vibs[vibid]
            _, _, name, version = vibid.split('_')
            vibName = "%s-%s.i386.vib" % (name, version)
            vibDir = os.path.join(pxedir, 'vibs')
            if not os.path.exists(vibDir):
               os.makedirs(vibDir)
            vib.WriteVibFile(os.path.join(vibDir, vibName))
            continue

         for payload, fobj in self.imageprofile.vibs[vibid].IterPayloads():
            if payload.payloadtype in (payload.TYPE_VGZ, payload.TYPE_TGZ,
                                       payload.TYPE_BOOT,
                                       payload.TYPE_INSTALLER_VGZ):
               # We assume GenerateVFATNames() is working properly, and that
               # vibstates is populated.
               state = self.imageprofile.vibstates[vibid]
               payloadfn = state.payloads[payload.name]
            else:
               payloadfn = payload.name

            if not payloadfn:
               msg = "VIB '%s' has payload with no name." % vibid
               raise Errors.VibFormatError(None, msg)

            if checkdigests and not payload.checksums:
               msg = ("Digest checking is enabled, but VIB payload does not "
                      "have a checksum.")
               raise Errors.VibPayloadDigestError(vibid, payload.name, msg)

            if checkdigests:
               try:
                  checksum = payload.GetPreferredChecksum()
                  algo = checksum.checksumtype.replace("-", "")
                  fobj = HashedStream(fobj, checksum.checksum, algo)
               except Exception as e:
                  msg = "Unable to calculate digest for VIB payload: %s." % e
                  raise Errors.VibPayloadDigestError(vibid, payload.name, msg)

            if payload.payloadtype in (payload.TYPE_VGZ, payload.TYPE_TGZ,
                                       payload.TYPE_BOOT_COM32_BIOS,
                                       payload.TYPE_BOOT_PXE_BIOS):
               newfpath = os.path.join(pxedir, payloadfn)
               self._CopyFileObjToFileName(fobj, newfpath, payload.size)
            elif (payload.payloadtype == payload.TYPE_INSTALLER_VGZ
                  and installer):
               newfpath = os.path.join(pxedir, payloadfn)
               self._CopyFileObjToFileName(fobj, newfpath, payload.size)
            elif payload.payloadtype == payload.TYPE_BOOT:
               imgpayldtar.AddPayload(fobj, payload, payloadfn)
               if checkdigests:
                  fobj.reset()
               else:
                  fobj.seek(0)
               newfpath = os.path.join(pxedir, payloadfn)
               self._CopyFileObjToFileName(fobj, newfpath, payload.size)

      imgpayldtar.close()
      imgpayldfobj.seek(0)

      if installer:
         newfpath = os.path.join(pxedir, self.PAYLOADTAR_NAME)
         self._CopyFileObjToFileName(imgpayldfobj, newfpath)

   def _AddDatabase(self, pxedir):
      # This method generates a tar database from the image profile, writes it
      # to a temp file, then adds the temp file to the PXE directory
      db = Database.TarDatabase()
      db.profiles.AddProfile(self.imageprofile)
      for vibid in self.imageprofile.vibIDs:
         if self.imageprofile.vibstates[vibid].boot:
            db.vibs.AddVib(self.imageprofile.vibs[vibid])

      # check if vib signatures must be saved
      savesig = self.imageprofile.IsSecureBootReady()

      try:
         tmpf = tempfile.TemporaryFile()
         db.Save(dbpath=tmpf, savesig=savesig)
         tmpf.seek(0, 0)
      except Errors.EsxupdateError:
         # Preserve stack trace to enable troubleshooting the root cause.
         raise
      except EnvironmentError as e:
         msg = "Could not create temporary database: %s." % e
         raise Errors.DatabaseIOError(None, msg)

      newfpath = os.path.join(pxedir, self.DATABASE_NAME)
      self._CopyFileObjToFileName(tmpf, newfpath)

   def _AddBootCfg(self, pxedir, installer=True, features=None):
      bootcfg = self._GetBootCfg(installer, features=features)
      assert bootcfg is not None,\
            "No module in image profile '%s'..." % (self.imageprofile.name)
      bootcfg.write(os.path.join(pxedir, 'boot.cfg'))

   def Write(self, pxedir, checkdigests=True, installer=True,
             checkacceptance=True, features=None):
      """Write out the files to a PXE directory.
            Parameters:
               * pxedir          - A string giving the absolute path to a
                                   directory.  Files for the PXE will be written
                                   to this directory.
               * checkdigests    - If True, payload digests will be verified
                                   when the PXE is written. Defaults to True.
               * installer       - Enable the installer in the booted image.
                                   Defaults to True.
               * checkacceptance - If True, validate the Acceptance Level of
                                   each VIB. If the validation fails, an
                                   exception is raised. Defaults to True.
               * features        - Features state switches to enable or disable
            Raises:
               * DatabaseIOError       - If unable to write the tar database to
                                         a temporary file.
               * ImageIOError          - If unable to write to a temporary file
                                         or the image output, or unable to
                                         compute the MD5 checksum of the image.
               * ProfileFormatError    - If the image profile has consistency
                                         errors.
               * VibDownloadError      - If unable to download one or more VIBs.
               * VibFormatError        - If one or more VIBs is not in proper
                                         VIB format.
               * VibIOError            - If unable to obtain the location of,
                                         or read data from, a VIB.
               * VibPayloadDigestError - If the calculated digest for one or
                                         more VIB payloads does not match the
                                         value given in VIB metadata.
      """

      self._CheckVibFiles(checkacceptance)
      self._DeployVib(pxedir, checkdigests, installer)
      self._AddDatabase(pxedir)
      self._AddBootCfg(pxedir, installer, features=features)

   def WriteRecord(self, name, recordFile, pxeDir, treeMD5,
                   installer, targetType, opts=None, features=None):
      """Write out a PXE record file for use by the pxe-boot perl script.
            Parameters:
               * name       - A name for the PXE image.
               * recordFile - The full path to the PXE record file that we wish
                              to write to.
               * pxeDir     - The full path to the directory that contains the
                              staged PXE files.
               * treeMD5    - An hashsum (of the path to your tree) that's used
                              to distinguish between your different trees.
               * installer  - Enables the installer in the PXE image.
               * targetType - The build type that we're using (obj, beta, release)
               * opts       - Any additional options that need to be passed to
                              the pxe-boot script.
               * features   - Feature state switches that need to be deployed
                              to the record.
      """
      syslinuxCount = 0
      imgCount = 0

      localOpts = opts.copy() or {}

      localOpts["pxetype"] = targetType

      if installer:
         localOpts["bootargs"] = "runweasel"
      if features:
         for feature in features:
            localOpts['FeatureState.%s' % feature] = features[feature]

      #
      # Add options of the form "syslinux.0", "syslinux.1"... to define syslinux
      # modules. Specify these in the config file as paths relative to the
      # config file we'll be writing.
      #

      # Internal deployed PXE will contain extra syslinux modules, add
      # them before scanning the vibs.
      pxeBootFiles = ['gpxelinux.0', 'ifgpxe.c32', 'ipxe-undionly.0',
                      'mboot.efi']
      for name in pxeBootFiles:
         localOpts['syslinux.%s' % syslinuxCount] = name
         syslinuxCount += 1

      for vibid in self.imageprofile.vibIDs:
         for payload, fobj, in self.imageprofile.vibs[vibid].IterPayloads():
            if payload.payloadtype in [payload.TYPE_BOOT_COM32_BIOS,
                                       payload.TYPE_BOOT_PXE_BIOS]:
               filePath = os.path.join(pxeDir, payload.name)
               relPath = os.path.relpath(filePath,
                                         os.path.dirname(recordFile))
               localOpts["syslinux.%s" % syslinuxCount] = relPath
               syslinuxCount += 1

      #
      # Add options of the form "image.0", "image.1".... to define
      # modules to be loaded.  Specify these in the config file as
      # paths relative to the config file we'll be writing.
      #
      payload_types = [Vib.Payload.TYPE_TGZ, Vib.Payload.TYPE_VGZ,
                       Vib.Payload.TYPE_BOOT]
      if installer:
         payload_types.append(Vib.Payload.TYPE_INSTALLER_VGZ)

      bootorder = self.imageprofile.GetBootOrder(payload_types)
      modules = [ p.localname for (vibid, p) in bootorder ]

      modules.append(self.DATABASE_NAME)
      if installer:
         modules.append(self.PAYLOADTAR_NAME)

      for m in modules:
         filePath = os.path.join(pxeDir, m)
         relPath = os.path.relpath(filePath,
                                   os.path.dirname(recordFile))
         localOpts["image.%s" % imgCount] = relPath
         imgCount += 1

      #
      # List other (non-boot) VIBs that are registered against this
      # image profile or copied in manually. In either case VIBs will
      # be present under vibs folder.
      #
      vibCount = 0
      vibDir = os.path.join(pxeDir, 'vibs')
      if os.path.exists(vibDir):
         for name in os.listdir(vibDir):
            localOpts["vib.%s" % vibCount] = 'vibs/' + name
            vibCount += 1

      #
      # The format for these files is:
      # <treeID>.<targetType>.<buildType>.key = val
      #
      output = ''
      for key in localOpts:
         output += "%s.%s.%s.%s = %s\n" % (treeMD5,
                                           targetType,
                                           localOpts['bldtype'],
                                           key,
                                           localOpts[key])

      with open(recordFile, 'w') as record:
         record.write(output)

