#!/usr/bin/python
########################################################################
# Copyright (C) 2010-2011, 2015-2016 VMWare, Inc.                      #
# All Rights Reserved                                                  #
########################################################################

import os
import shutil
import tarfile
import tempfile

from vmware.esximage import Downloader, Errors, Vib
from vmware.esximage.Utils import BootCfg

"""This module provides an ancestor class for creating different image types
   based on an image profile."""

class ImageBuilder(object):
   """This class is a skeleton to be inherited for the different methods of
   generating an image with the contents of an image profile."""

   DATABASE_NAME = "imgdb.tgz"
   PAYLOADTAR_NAME = "imgpayld.tgz"

   def __init__(self, imageprofile):
      """Class constructor.
            Parameters:
               * imageprofile - An instance of ImageProfile. The 'vibs'
                                attribute of the object must contain valid
                                references to the VIBs in the 'vibIDs'
                                property. Those references must include either
                                a file object or a valid remote location.
      """
      self.imageprofile = imageprofile

   @staticmethod
   def _GetSeekableFileObj(url):
      # Each VIB can potentially contain more than one payload. It is likely
      # that the order of payloads within the VIB will not be the order in
      # which they will be written to the target. Therefore, the file-like
      # object from which a VIB's payloads are read must be seekable. This
      # method takes a URL or path as input and returns a seek-able file object.
      # For remote URLs, it accomplishes that by downloading to a temporary
      # file.
      d = Downloader.Downloader(url).Open()
      # We only care if something is seekable. Let Downloader figure out
      # whether it's local or remote.
      if hasattr(d, "seek"):
         return d
      # Else, copy it locally.
      t = tempfile.TemporaryFile()
      data = d.read(512)
      while data:
         t.write(data)
         data = d.read(512)
      d.close()
      t.seek(0)
      return t

   @classmethod
   def _CheckVibFile(cls, vib, checkacceptance=True):
      # This method just ensures that a single VIB has an appropriate file-like
      # object behind it.

      # Should probably make public attributes/properties/methods to implement
      # this check.
      if vib._arfile is not None and vib._arfile._haveseek:
         pass
      elif not vib.remotelocations:
         msg = ("VIB %s has neither a seek-able file object nor a URL "
                "location. This may indicate a problem with the depot "
                "metadata. " % vib.id)
         raise Errors.VibIOError(None, msg)
      else:
         problems = list()
         success = False
         for url in vib.remotelocations:
            try:
               vib.OpenFile(cls._GetSeekableFileObj(url))
               # We only need one. :-)
               success = True
               break
            except Exception as e:
               problems.append(str(e))

         if not success:
            msg = ("Error retrieving file for VIB '%s': %s."
                   % (vib.id, "; ".join(problems)))
            raise Errors.VibDownloadError(None, None, msg)

      if checkacceptance:
         vib.VerifyAcceptanceLevel()

   def _CheckVibFiles(self, checkacceptance=True):
      # This method makes sure that every VIB has an appropriate file object
      # behind it, so that we'll be able to read payloads into the target object.
      # It also checks to make sure that things in vibIDs have corresponding
      # objects in vibs. This is the only place where we make this check, so
      # this method should be called before _AddPayloads() or _AddDatabase().
      for vibid in self.imageprofile.vibIDs:
         try:
            vib = self.imageprofile.vibs[vibid]
         except KeyError:
            msg = "Could not find object for VIB '%s'." % vibid
            raise Errors.ProfileFormatError(self.imageprofile.name, msg)
         self._CheckVibFile(vib, checkacceptance)

   def _GetBootCfg(self, installer=True, moduleroot='', features=None, isoImage=False):
      '''Return BootCfg instance if boot modules is not zero, otherwise return
         None
         Parameters:
            * installer  - True if the bootcfg is for installer
            * moduleroot - root for module files
            * native     - True if the bootcfg contains only native drivers
      '''
      payload_types = [Vib.Payload.TYPE_TGZ, Vib.Payload.TYPE_VGZ,
                       Vib.Payload.TYPE_BOOT]
      if installer:
         payload_types.append(Vib.Payload.TYPE_INSTALLER_VGZ)

      bootorder = self.imageprofile.GetBootOrder(payload_types)

      modules = [p.localname for (vibid, p) in bootorder]
      if not modules:
         return None

      bootcfg = BootCfg.BootCfg()

      # ESX Database
      modules.append(self.DATABASE_NAME)

      if installer:
         bootcfg.kernelopt["runweasel"] = None
         modules.append(self.PAYLOADTAR_NAME)
         bootcfg.title = "Loading ESXi installer"
      else:
         bootcfg.kernelopt["autoPartition"] = "TRUE"
         bootcfg.title = "Loading ESXi"

      if isoImage:
         bootcfg.kernelopt["cdromBoot"] = None

      if features:
         for feature in features:
            bootcfg.kernelopt['FeatureState.%s' % feature] = features[feature]

      if moduleroot:
         modules = [os.path.join(moduleroot, module) for module in modules]

      bootcfg.kernel = modules[0]

      bootcfg.modules = modules[1:]

      return bootcfg

   def _AddPayloads(self, target, checkdigests=True):
      # This method adds each payload to <target>, while doing the right things
      # for various payload types.
      raise NotImplementedError("_AddPayloads is not implemented in the child"
                                " class.")

   def _AddDatabase(self, target):
      # This method generates a tar database from the image profile, writes it
      # to a temp file, then adds the temp file to <target>.
      raise NotImplementedError("_AddDatabase is not implemented in the child"
                                " class.")

   def _AddMetadataZip(self, target):
      # This method generates a metadata.zip from the image profile.
      raise NotImplementedError("_AddMetadataZip is not implemented in the"
                                " child class.")

   def _AddProfileXml(self, target):
      # Adds profile.xml to the upgrade directory in the target.
      raise NotImplementedError("_AddProfileXml is not implemented in the"
                                " child class.")

   # Not necessary for a pxeboot, but could be generated either way just incase
   # it is needed.
   def _AddBootCfg(self, target, installer=True, features=None):
      # This method populates efi/boot/boot.cfg. It is required for EFI.
      raise NotImplementedError("_AddBootCfg is not implemented in the child"
                                " class")

