#!/usr/bin/python
########################################################################
# Copyright (C) 2010-2018 VMWare, Inc.                                 #
# All Rights Reserved                                                  #
########################################################################

import logging
import os
import shutil
import gzip
import sys
from socket import gethostname

from vmware.runcommand import runcommand, RunCommandError

from .. import Database
from .. import Errors
from .. import ImageProfile
from .. import Vib
from . import Installer
from ..Utils import HashedStream
from ..Utils import HostInfo
from ..Utils import LockFile
from ..Utils import EsxGzip
from ..Utils import Ramdisk
from ..Utils.Misc import byteToStr

import time, errno

BACKUP_SH = '/sbin/backup.sh'
SECURE_MOUNT_SCRIPT = '/usr/lib/vmware/secureboot/bin/secureMount.py'

class InvalidBootcfgError(Exception):
   pass

class InvalidBootbankError(Exception):
   pass

log = logging.getLogger(os.path.basename(__file__))

class Bootcfg(object):
   """Bootcfg encapsulates the boot.cfg file of a bootbank.

      Class Variables:
         * BOOTSTATE_SUCCESS   - The bootbank has booted successfully.
         * BOOTSTATE_UPDATED   - The bootbank is upgraded, but not booted yet.
         * BOOTSTATE_ATTEMPTED - The bootbank is being booted, but not finish
                                 yet.
         * BOOTSTATE_EMPTY     - The bootbank is empty.
         * BOOTSTATE_STAGED    - New IP has been staged to bootbank, but update
                                 is not finished yet.

      Attributes:
         * kernel    - Kernel name
         * kernelopt - The options to kernel module
         * modules   - List of modules to load. The list is constructed and
                       ordered by ImageProfile class.
         * build     - Version info of core ESX module, usually in
                       'version-build' format.
         * updated   - The updated sequential number.
         * bootstate - Bootbank state, one of BOOTSTATE_TYPES.
   """
   BOOTSTATE_SUCCESS   = 0
   BOOTSTATE_UPDATED   = 1
   BOOTSTATE_ATTEMPTED = 2
   BOOTSTATE_EMPTY     = 3
   BOOTSTATE_STAGED    = 4
   BOOTSTATE_TYPES = (BOOTSTATE_SUCCESS, BOOTSTATE_UPDATED,
         BOOTSTATE_ATTEMPTED, BOOTSTATE_EMPTY, BOOTSTATE_STAGED)
   INSTALLER_BOOTSTATES = (BOOTSTATE_EMPTY, BOOTSTATE_STAGED, BOOTSTATE_UPDATED)

   def __init__(self, path):
      """
         Parameters:
            * path - File path to boot.cfg.
      """
      self._path = path
      self.Clear()

   path = property(lambda self: self._path)

   def Clear(self):
      """Mark the bootbank as empty."""
      self.kernel = ''
      self.kernelopt = ''
      self.modules = []
      self.build = ''
      self.updated = -1
      self.bootstate = self.BOOTSTATE_EMPTY
      self.title = ''
      self.prefix = ''
      self.timeout = 5

   def Load(self):
      """Load and parse boot.cfg file.
         Exceptions:
            * InvalidBootcfgError - If not all of the expected keys are loaded,
                                   an error in parsing the value, or an error to
                                   open the file.
      """
      self.Clear()
      expectedkeys = ['updated', 'bootstate', 'kernel', 'kernelopt', 'modules',
            'build']
      try:
         for line in open(self._path):
            line = line.strip()
            if line:
               k, sep, v = line.partition('=')
               if sep != '=':
                  continue

               k = k.strip()
               v = v.strip()
               if k == 'updated':
                  self.updated = int(v)
               elif k == 'bootstate':
                  self.bootstate = int(v)
               elif k == 'kernel':
                  self.kernel = v
               elif k == 'kernelopt':
                  self.kernelopt = v
               elif k == 'modules':
                  self.modules = v.split(' --- ')
               elif k == 'build':
                  self.build = v
               elif k == 'title':
                  self.title = v
               elif k == 'prefix':
                  self.prefix = v
               elif k == 'timeout':
                  self.timeout = int(v)
               else:
                  log.info('Unrecognized value "%s" in boot.cfg' % (line))

               if k in expectedkeys:
                  expectedkeys.remove(k)
      except Exception as e:
         self.Clear()
         msg = 'Error parsing bootbank boot.cfg file %s: %s' % (self.path, e)
         raise InvalidBootcfgError(msg)

      if expectedkeys:
         self.Clear()
         msg = '%s value is expected in boot.cfg %s, but not found' % (
               expectedkeys, self._path)
         raise InvalidBootcfgError(msg)

   def Save(self):
      """Save Bootcfg instance content to file.
         Exceptions:
            * InstallationError - If there is an error in saving Bootcfg to file.
      """
      try:
         fp = open(self.path + '.tmp', 'w')
         c = ['bootstate=%s' % (str(self.bootstate)),
            'title=%s' % (self.title),
            'timeout=%s' % (str(self.timeout)),
            'prefix=%s' % (self.prefix),
            'kernel=%s' % (self.kernel),
            'kernelopt=%s' % (self.kernelopt),
            'modules=%s'% (' --- '.join(self.modules)),
            'build=%s' % (self.build),
            'updated=%s' % (str(self.updated))]
         fp.write('\n'.join(c))
         fp.write('\n')
         fp.close()
         os.rename(self.path + '.tmp', self.path)
      except EnvironmentError as e:
         msg = 'Failed to save Bootcfg to file %s: %s' % (self._path, e)
         raise Errors.InstallationError(None, msg)

   def UpdateState(self, state, liveimage=False):
      """Change bootstate value to BOOTSTATE_EMPTY, BOOTSTATE_STAGED or
         BOOTSTATE_UPDATED, which are managed by Installer.

            Parameters:
               * state - one of BOOTSTATE_EMPTY, BOOTSTATE_STAGED,
                         BOOTSTATE_UPDATED, BOOTSTATE_SUCCESS (for live image)
            Exceptions:
               * ValueError - If the given state is not managed by Installer.

      """
      if state in self.INSTALLER_BOOTSTATES:
         log.info("%s: bootstate changed from %s to %s"
            % (self._path, str(self.bootstate), str(state)))
         self.bootstate = state
      else:
         if liveimage and state == self.BOOTSTATE_SUCCESS:
            log.info("%s: bootstate changed from %s to %s"
               % (self._path, str(self.bootstate), str(state)))
            self.bootstate = state
         else:
            msg = 'Can not change state to %s, only %s are allowed' % (
                  state, self.INSTALLER_BOOTSTATES)
            raise ValueError(msg)

class BootBank(object):
   """Encapsulates attributes of a single bootbank.

      Class Variables:
         * DB_FILE  - File name of bootbank package database.
         * BOOT_CFG - File name of the config file for bootloader

      Attributes:
         * db      - An instance of Database.TarDatabase representing the
                     bootbank package database.
         * updated - "updated" serial number from boot.cfg.  If the bootbank
                     contains an invalid image or boot.cfg, this should be -1.
         * path    - The full file path for the mounted bootbank.
   """
   DB_FILE = 'imgdb.tgz'
   BOOT_CFG = 'boot.cfg'

   def __init__(self, bootbankpath):
      """
         Exceptions:
            InvalidBootbankError - thrown if a valid bootbank could not be found
                              at the bootbankpath
      """
      if not os.path.isdir(bootbankpath):
         raise InvalidBootbankError("%s is not a directory!" % (bootbankpath))

      self._path = bootbankpath
      dbpath = os.path.join(bootbankpath, self.DB_FILE)
      # Load existing database or create a new in-ram database, do not commit
      # to the disk/folder.
      self.db = Database.TarDatabase(dbpath, dbcreate=False)
      bootcfgpath = os.path.join(bootbankpath, self.BOOT_CFG)
      self.bootcfg = Bootcfg(bootcfgpath)

   def __str__(self):
      return "<BootBank:%s>" % (self.path)

   @property
   def path(self):
      return self._path

   def _getupdated(self):
      return self.bootcfg.updated

   def _setupdated(self, updated):
      self.bootcfg.updated = updated

   updated = property(_getupdated, _setupdated)

   @property
   def bootstate(self):
      return self.bootcfg.bootstate

   def _UpdateVib(self, newvib):
      """Update missing properties of vib metadata

         Parameters:
            * newvib   - The new vib to use as source
         Returns:
            None if the update succeeds, Exception otherwise
         Exceptions:
            VibMetadataError
      """
      try:
         self.db.vibs[newvib.id].SetSignature(newvib.GetSignature())
         self.db.vibs[newvib.id].SetOrigDescriptor(newvib.GetOrigDescriptor())
      except Exception as e:
         raise Errors.VibMetadataError(e)

   def _AcquireLock(self, lockfile, timeout=10):
      '''Lock a filelock with retries'''
      flock = LockFile.LockFile(lockfile)
      starttime = time.time()
      while True:
         try:
            flock.Lock()
            break
         except LockFile.LockFileError as e:
            if (time.time() - starttime) >= self.timeout:
               raise LockFile.LockFileException('LockFile %s timed out ' \
                                                'after %s seconds.' %
                                                (lockfile, str(timeout)))
            # check again in 1 second
            time.sleep(1)
      return flock

   def _ReleaseLock(self, flock):
      '''Unlock a filelock and delete file'''
      flock.Unlock()
      os.unlink(flock._lockfile)

   def Load(self, raiseerror=True, createprofile=False):
      """Load database and boot.cfg from the bootbank
         Exceptions:
            * InvalidBootbankError - If raiseerror is True and there is an error
                                     loading boot.cfg or database.
      """
      try:
         self.bootcfg.Load()
      except InvalidBootcfgError as e:
         msg = 'Error in loading boot.cfg from bootbank %s: %s' % \
               (self._path, e)
         if raiseerror:
            raise InvalidBootbankError(msg)
         else:
            log.warn('Ignoring error when loading bootbank: %s' % msg)

      if self.bootcfg.bootstate in (Bootcfg.BOOTSTATE_UPDATED,
            Bootcfg.BOOTSTATE_STAGED, Bootcfg.BOOTSTATE_SUCCESS):
         try:
            self.db.Load()
         except (Errors.DatabaseFormatError, Errors.DatabaseIOError) as e:
            self.db.Clear()
            self.bootcfg.UpdateState(Bootcfg.BOOTSTATE_EMPTY)
            msg = 'Error in loading database for bootbank %s: %s' % \
                  (self._path, e)
            if raiseerror:
               raise InvalidBootbankError(msg)
            else:
               log.warn('Ignoring error when loading bootbank: %s' % msg)

         if self.db.profile is None and createprofile:
            log.debug('Creating an empty ImageProfile for bootbank %s' % \
                      self._path)
            self.db.Clear()
            pname = 'Empty ImageProfile'
            creator = gethostname()
            p = ImageProfile.ImageProfile(pname, creator)
            self.db.profiles.AddProfile(p)

         if self.db.profile is not None:
            self.db.profile.vibs = self.db.vibs

   def Clear(self):
      """Clear bootbank DB to empty and flag the bootbank as empty
         Caller is responsible to call Save to make the change persistent.
      """
      self.bootcfg.UpdateState(self.bootcfg.BOOTSTATE_EMPTY)
      self.db.Clear()

   def Save(self):
      """Flush bootcfg and database to persistent storage.
      """
      self.db.Save(savesig=True)
      self.bootcfg.Save()

   def ClearContent(self):
      """Mark the bootbank as empty in database and boot.cfg. Files other than
         database and boot.cfg are removed as well.
      """
      try:
         self.Clear()
         self.Save()
      except Exception as e:
         msg = 'Error in clearing bootcfg or DB for bootbank %s: %s' % (
               self._path, e)
         raise Errors.InstallationError(None, msg)

      lockfile = '/tmp/%s.lck' % self._path.replace("/", "_")
      flock = self._AcquireLock(lockfile)
      try:
         for root, dirs, files in os.walk(self.path, topdown=True):
            for dn in dirs:
               shutil.rmtree(os.path.join(root, dn))
            for name in files:
               if name not in (BootBank.BOOT_CFG, BootBank.DB_FILE):
                  os.unlink(os.path.join(root, name))
      except EnvironmentError as e:
         msg = 'Failed to clear bootbank content %s: %s' % (self._path, e)
         raise Errors.InstallationError(None, msg)
      finally:
         self._ReleaseLock(flock)

   def FileCopy(self, srcbank):
      """Copy all srcbank files to this bootbank."""
      lockfile = '/tmp/%s.lck' % self._path.replace("/", "_")
      flock = self._AcquireLock(lockfile)
      try:
         # Remove all files of this bootbank
         rc, _ = runcommand("rm -rf %s/*" % self.path)
         # Copy all files from srcbank to this bootbank
         rc, _ = runcommand("cp %s/* %s/" % (srcbank.path, self.path))
      except RunCommandError as e:
         msg = 'Failed to copy files from %s to %s: %s' % (
                 srcbank.path, self._path, e)
         raise Errors.InstallationError(None, msg)
      finally:
         self._ReleaseLock(flock)

   def Verify(self):
      """ Verify whether the content of bootbank is consistent with imgprofile
      in DB.
      """
      #TODO: verify payloads according to DB

      #Verify vib payload

   def Stage(self, srcbank, keeps, imgprofile):
      """Stage the initial content of this bootbank. If srcbank is different
         from this bootbank, for VIBs in keeps list, payloads will be
         copied to this bootbank and renamed according to imgprofile. If
         srcbank is this bootbank, VIBs not in keeps list will be removed from
         the bootbank and the VIBs in keeps list will be renamed according to target
         image profile.

         Parameters:
            * srcbank    - BootBank instance, either 'bootbank' or
                           'altbootbank', the content is used to build target
                           image profile as source input.
            * keeps      - List of id of VIBs which need to be kept from srcbank
                           to construct target imageprofile.
            * imgprofile - Target image profile for this bootbank.

         Exceptions:
            * InstallationError - If there is an error to stage the imgprofile
                                  to this bootbank
      """
      if srcbank.path == self.path:
         self._inplaceStage(keeps, imgprofile)
      else:
         self._copyStage(srcbank, keeps, imgprofile)

      # Database is updated to the target ImageProfile, though the content is
      # not ready yet. It is fine because this bootbank is considered as
      # empty after calling stage.
      self.db.vibs.clear()
      self.db.vibs = imgprofile.vibs
      self.db.profiles.clear()
      self.db.profiles.AddProfile(imgprofile)

   def _inplaceStage(self, keeps, imgprofile):
      #NOTE: This only uses current bootbank VIB content, we might do better in some
      # case to use VIBs from /bootbank.
      self.bootcfg.UpdateState(self.bootcfg.BOOTSTATE_EMPTY)
      self.bootcfg.Save()
      kept = set()

      for vibid in self.db.profile.vibstates:
         # In place renaming to a temporary name
         vibstate = self.db.profile.vibstates[vibid]
         if vibid in keeps:
            deststate = imgprofile.vibstates[vibid]
            for pl in self.db.vibs[vibid].payloads:
               if pl.payloadtype not in BootBankInstaller.SUPPORTED_PAYLOADS:
                  continue

               plname = pl.name
               srcname = None
               if plname in vibstate.payloads:
                  srcname = vibstate.payloads[plname]

               if srcname is not None:
                  destname = deststate.payloads[plname] + '.stg'
                  dst = os.path.join(self._path, destname)
                  src = os.path.join(self._path, srcname)

                  try:
                     os.rename(src, dst)
                  except OSError as e:
                     msg = 'Failed to '
                     raise Errors.InstallationError(str(imgprofile.vibIDs),
                           msg)
               else:
                  msg = ('Can not locate source for payload %s of VIB %s '
                         ' from bootbank %s' % (plname, vibstate.id, self._path))
                  raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

               vibstate.payloads[plname] = destname
            kept.add(vibstate.id)
         else:
            # Remove unwanted vibs
            log.info('Removing VIB %s from %s' % (vibid, self._path))
            for plname in vibstate.payloads:
               localfile = os.path.join(self.path, vibstate.payloads[plname])
               if not os.path.isfile(localfile):
                  continue

               log.info('Removing payload %s at %s' % (plname, localfile))
               try:
                  os.unlink(localfile)
               except EnvironmentError as e:
                  msg = ('Can not remove payload %s in VIB %s: %s' %
                        (localfile, vibid, e))
                  raise Errors.InstallationError(None, msg)
            del self.db.vibs[vibid]

      if kept != set(keeps):
         left = set(keeps) - kept
         msg = 'VIBs %s are not found in bootbank %s' % (left, self._path)
         raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

      # re-establish local filename
      # In order to avoid name collision between original localname and
      # target localname, payload is renamed to a name space ends with
      # '.stg' and then move to the final name.
      for vibid in keeps:
         vibstate = self.db.profile.vibstates[vibid]
         for pl in self.db.vibs[vibid].payloads:
            if pl.payloadtype not in BootBankInstaller.SUPPORTED_PAYLOADS:
               continue

            plname = pl.name
            src = os.path.join(self.path, vibstate.payloads[plname])
            dst = src[:-4]
            if src.endswith('.stg'):
               try:
                  os.rename(src, dst)
               except OSError as e:
                  msg = ('Failed to rename temporary payload file %s '
                          'to final name %s: %s.' % (src, dst, e))
                  raise Errors.InstallationError(str(imgprofile.vibIDs), msg)
            else:
               msg = 'Unexpected temporary payload file name %s' % (src)
               raise Errors.InstallationError(str(imgprofile.vibIDs), msg)
            vibstate.payloads[plname] = vibstate.payloads[plname][:-4]

   def _copyStage(self, srcbank, keeps, imgprofile):
      # Preparation altbootbank based on srcbank
      # TODO: we can do better if there is a partial databse
      # Let's try the hard way first - just copy src from bootbank
      self.ClearContent()
      for vibid in keeps:
         if vibid not in srcbank.db.profile.vibIDs:
            msg = 'Can not find VIB %s in bootbank %s.' % (vibid,
                  srcbank.path)
            raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

         log.info('Copying payloads of %s from %s to %s' % (vibid,
            srcbank.path, self._path))
         deststate = imgprofile.vibstates[vibid]
         vibstate = srcbank.db.profile.vibstates[vibid]
         for pl in srcbank.db.vibs[vibid].payloads:
            if pl.payloadtype not in BootBankInstaller.SUPPORTED_PAYLOADS:
               continue

            plname = pl.name
            srcfile = None
            if plname in vibstate.payloads:
               srcfile = os.path.join(srcbank.path, vibstate.payloads[plname])

            if srcfile is not None:
               dstfile = os.path.join(self.path, deststate.payloads[plname])
               log.info('Copying %s to %s' % (srcfile, dstfile))
               try:
                  shutil.copy2(srcfile, dstfile)
               except EnvironmentError as e:
                  msg = 'Error in copying from %s to %s: %s' % (srcfile,
                        dstfile, e)
                  raise Errors.InstallationError(str(imgprofile.vibIDs), msg)
               #TODO: verify copy
               #log.debug('Verifying the file %s' % (dstfile))
            else:
               msg = ('Can not locate source for payload %s of VIB %s '
                      'from bootbank %s' % (plname, vibid, srcbank.path))
               raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

class BootBankInstaller(Installer):
   """BootBankInstaller is the Installer class for building
      a ESXi VFAT bootbank image.

      Class Variables:
         * BOOTBANK_STATE_FRESH   - altbootbank is either empty or a previous boot
                                    image.
         * BOOTBANK_STATE_STAGED  - altbootbank is staged with new ImageProfile,
                                    and VIBs has been installed to the bootbank,
                                    but the host will still boot from current
                                    bootbank in next reboot.
         * BOOTBANK_STATE_UPDATED - altbootbank is upgraded. The host will boot
                                    from altbootbank in next reboot.
         * SUPPORTED_PAYLOADS     - tuple of payload types that will be handled
                                    by the installer.

      Attributes:
         * bootbank    - A BootBank instance representing /bootbank, the bank
                         from which the host booted
         * altbootbank - A BootBank instance representing /altbootbank, the
                         bank holding the future (or previous) boot image
         * stagebootbank - A BootBank instance representing STAGEBOOTBANK,
                           the ramdisk holding staged boot image.
         * bootbankstate - Indicate whether altbootbank is staged and / or
                           updated, combination of BOOTBANK_STATE_* bit.
   """
   installertype = "boot"
   priority = 10
   BOOTBANK = '/bootbank'
   ALTBOOTBANK = '/altbootbank'
   STAGEBOOTBANK = '/tmp/stagebootbank'
   STAGEBOOTBANK_NAME = 'stagebootbank'
   STAGEBOOTBANK_SIZE = 250
   BUFFER_SIZE = 1024 * 1024

   # NOTE: 10MB for database, state.tgz
   BOOTBANK_PADDING_MB = 10

   # BOOTBANK_STATE should be considered as bit pattern state
   # because we can have staged and updated state.
   # The first bit ( int 1) is for staged or not.
   # The second bit ( int 2 ) is for updated or not.
   # 'staged' and 'updated' state is that the both bits are on:
   # > BOOTBANK_STATE_STAGED | BOOTBANK_STATE_UPDATED
   # We can check whether the stage is 'staged',
   # > state & BOOTBANK_STATE_STAGED
   # To check whether it's updated,
   # > state & BOOTBANK_STATE_UPDATED
   BOOTBANK_STATE_FRESH = 0
   BOOTBANK_STATE_STAGED = 1
   BOOTBANK_STATE_UPDATED = 2

   SUPPORTED_VIBS = set([Vib.BaseVib.TYPE_BOOTBANK,])
   SUPPORTED_PAYLOADS = (Vib.Payload.TYPE_TGZ, Vib.Payload.TYPE_VGZ,
         Vib.Payload.TYPE_BOOT)

   def __init__(self, bootbank=BOOTBANK, altbootbank=ALTBOOTBANK):
      """Constructor for this Installer class.
         /bootbank and /altbootbank must both exist. /bootbank must points to
         valid bootbank with boot.cfg file.  If not, this class is not
         instantiated.

         Exceptions:
            InstallerNotAppropriate - Environment is not appropriate for
               this type of installer.
      """
      #TODO: add probe to host environment, raise error if bootbank is not
      # supported.
      try:
         self.bootbank = BootBank(bootbank)
      except InvalidBootbankError as e:
         msg = 'bootbank is invalid: %s' % (e)
         raise Errors.InstallerNotAppropriate(self.installertype, msg)

      try:
         self.bootbank.Load(createprofile = True)
         self.altbootbank = BootBank(altbootbank)
         # Altbootbank is not always available.
         self.altbootbank.Load(raiseerror = False)
         if os.path.exists(self.STAGEBOOTBANK):
            self.stagebootbank = BootBank(self.STAGEBOOTBANK)
            self.stagebootbank.Load(raiseerror = False)
         else:
            self.stagebootbank = None
      except InvalidBootbankError as e:
         msg = 'altbootbank is invalid: %s' % (e)
         raise Errors.InstallationError(None, msg)
      self.problems = list()

   @property
   def bootbankstate(self):
      staged = 0
      if self.stagebootbank and \
            self.stagebootbank.bootstate & Bootcfg.BOOTSTATE_STAGED:
         staged = self.BOOTBANK_STATE_STAGED
      if self.bootbank.updated < self.altbootbank.updated and \
            self.altbootbank.bootstate in (
            Bootcfg.BOOTSTATE_SUCCESS, Bootcfg.BOOTSTATE_UPDATED):
         # When a live vib is installed, updated is incremented,
         # but bootstate remains to 0 (BOOTSTATE_SUCCESS).
         return self.BOOTBANK_STATE_UPDATED | staged
      return self.BOOTBANK_STATE_FRESH | staged

   @property
   def currentbootbank(self):
      if self.bootbankstate & self.BOOTBANK_STATE_UPDATED:
         return self.altbootbank
      else:
         return self.bootbank

   @property
   def database(self):
      return self.currentbootbank.db

   @property
   def stagedatabase(self):
      if self.stagebootbank and \
            self.stagebootbank.bootstate & Bootcfg.BOOTSTATE_STAGED:
         return self.stagebootbank.db
      else:
         return None

   def CheckInstallationSize(self, imgprofile):
      '''Calculate the installation size of imgprofile on bootbank.
         Verify bootbank can hold the image. BOOTBANK_PADDING_MB is reserved for
         state.tgz and other extra files.

         NOTE: No attempt is made to estimate the size of state.tgz, transaction
               might still fail if state.tgz needs more than the reserved padding
               space.

         Parameters:
            * imgprofile - Target image profile

         Exceptions:
            * InstallationError - Installation size of this image profile can't
                                  fit in bootbank partition.
      '''
      totalsize = self.GetInstallationSize(imgprofile)
      stat = os.statvfs(self.altbootbank.path)
      capacity = (stat.f_bsize * stat.f_blocks -
                  self.BOOTBANK_PADDING_MB * 1024 *1024)
      if totalsize > capacity:
         msg = ('The pending transaction requires %d MB free space, however '
                'the maximum supported size is %d MB.' % (totalsize/1024/1024,
                   capacity/1024/1024))
         log.error(msg)
         raise Errors.InstallationError('', msg)

   def CheckBootState(self):
      '''Check the bootstate of current bootbank. If bootstate is not
         BOOTSTATE_SUCCESS, raise InstallationError exception.
      '''
      if (self.bootbank.bootstate != Bootcfg.BOOTSTATE_SUCCESS):
         msg = ('Current bootbank %s is not verified and most likely a '
                'serious problem was encountered during boot, it is not safe '
                'to continue altbootbank install. bootstate is %d, expected '
                'value is %s.' % ( self.bootbank.path, self.bootbank.bootstate,
                 Bootcfg.BOOTSTATE_SUCCESS))
         raise Errors.InstallationError('', msg)

   def PreInstCheck(self, imgprofile):
      '''Perform sanity check for new image profile change.
         Parameters:
            * imgprofile - Target image profile

         Exceptions:
            * InstallationError - /bootbank is not in BOOTSTATE_SUCCESS state or
                                  installation size can't fit in bootbank
                                  partition.
      '''
      self.CheckBootState()
      self.CheckInstallationSize(imgprofile)

   def _SetupStageBootbank(self):
      """Set self.stagebootbank."""
      # Check if the stage bootbank dir exist.
      if not os.path.exists(self.STAGEBOOTBANK):
         try:
            Ramdisk.CreateRamdisk(self.STAGEBOOTBANK_SIZE, \
                                  self.STAGEBOOTBANK_NAME, self.STAGEBOOTBANK)
         except Errors.InstallationError as e:
            # we need to set stagebootbank None when ramdisk create fail
            self.stagebootbank = None
            raise
      self.stagebootbank = BootBank(self.STAGEBOOTBANK)

   def _CleanupStageBootbank(self):
      """Empty stagebootbank and remove ramdisk."""
      Ramdisk.RemoveRamdisk(self.STAGEBOOTBANK_NAME, self.STAGEBOOTBANK)
      self.stagebootbank = None

   def StartTransaction(self, imgprofile, preparedest=True, **kwargs):
      """Initiates a new installation transaction. Calculate what actions
         need to be taken.  Prepare /altbootbank by copying all contents
         from /bootbank.
         This method may change the installation destination.

         Parameters:
            * imgprofile - The ImageProfile instance representing the
                           target set of VIBs for the new image
            * preparedest - Boolean, if True, then prepare the destination.
                           Set to false for a "dry run", to avoid changing
                           the destination.
            * forcebootbank - Boolean, if True, skip install of live image
                           even if its eligible for live install
         Returns:
            A tuple (installs, removes, staged), installs and removes are list
            of VIB IDs for HostImage.Stage() to install to the destination and
            to remove from the destination, in order to make it compliant
            with imgprofile. If altbootbank has already staged the imgprofile,
            staged is True.
            If there is nothing to do, (None, None, False) is returned.
         Exceptions:
            InstallationError
      """
      if not self.stagebootbank:
         self._SetupStageBootbank()
      try:
         self.stagebootbank.Load(raiseerror=False)
      except:
         self.stagebootbank.Clear()

      imgprofile = imgprofile.Copy()
      unsupported = imgprofile.vibIDs - self.GetSupportedVibs(imgprofile.vibs)
      for vibid in unsupported:
         imgprofile.RemoveVib(vibid)

      staged = False
      if (self.stagebootbank.bootstate == Bootcfg.BOOTSTATE_STAGED):
         if self.stagebootbank.db.profile is not None and \
            self.stagebootbank.db.profile.vibIDs == imgprofile.vibIDs:
            staged = True
         elif preparedest:
            # stagebootbank exists but it's not for the current transaction.
            # Need to clear it
            self.stagebootbank.ClearContent()

      adds, removes = imgprofile.Diff(self.database.profile)

      if staged:
         return (adds, removes, staged)

      if preparedest and (removes or adds):
         if self.bootbankstate & self.BOOTBANK_STATE_UPDATED:
            srcbootbank = self.altbootbank
         else:
            srcbootbank = self.bootbank

         keeps = set(imgprofile.vibIDs) - set(adds)
         self.stagebootbank.Stage(srcbootbank, keeps, imgprofile)

      return (adds, removes, staged)

   def UpdateVibDatabase(self, newvib):
      """Update missing properties of vib metadata
         New vibs are always installed in altbootbank

         Parameters:
            * newvib   - The new vib to use as source
         Returns:
            None if the update succeeds, Exception otherwise
         Exceptions:
            VibMetadataError
      """
      try:
         self.stagebootbank._UpdateVib(newvib)
      except Exception as e:
         log.debug('failed updating bootbank for %s', newvib.name)
         raise

   def VerifyPayloadChecksum(self, vibid, payload):
      """Verify the checksum of a given payload.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
         Returns:
            None if verification succeeds, Exception otherwise
         Exceptions:
            ChecksumVerificationError
            InstallationError
      """
      if payload.payloadtype in self.SUPPORTED_PAYLOADS:
         # Special case for payloads that encapsulate state.
         # These payloads can change. Hence their checksums
         # will not match the ones in the descriptor.
         # TODO: Since OEMs can add additional useropts modules,
         # we will need to get this list from the kernel.
         exemptPayloads = ["useropts", "features", "jumpstrt"]
         if any(payload.name.startswith(p) for p in exemptPayloads):
            return

         checksumfound = False
         for checksum in payload.checksums:
            if checksum.checksumtype == "sha-1":
               checksumfound = True
               try:
                  # The vib must be present in the currentbootbank
                  currentbootbank = self.currentbootbank
                  if vibid in currentbootbank.db.profile.vibIDs:
                     vibstate = currentbootbank.db.profile.vibstates[vibid]
                     rootpath = currentbootbank.path
                  else:
                     msg = 'Cannot find VIB %s in BootbankInstaller' % vibid
                     raise Errors.InstallationError(None, msg)

                  filepath = None
                  if payload.name in vibstate.payloads:
                     filepath = os.path.join(rootpath, vibstate.payloads[payload.name])
                  else:
                     msg = ("VIB %s found in BootbankInstaller (%s), but payload %s is "
                            "not found in the VIB" % (vibid, rootpath, payload.name))
                     raise Errors.InstallationError(msg)

                  # In a Bootbank installed host, the bootbank payloads for
                  # tardisks are generated by reading the tardisk node in the
                  # tardisks directory and gzip'ing the contents. This compressed
                  # payload has a different hash than the one in the original
                  # descriptor file. This is because the gzip algorithm used at
                  # build time adds timestamp and other file metadata to the
                  # compressed version. However, the sha-1 hash currently stored
                  # in the descriptor is a hash on the uncompressed payload.
                  # Hence we calculate the hash on the uncompressed version of
                  # the payload.
                  if os.path.isfile(filepath):
                     with gzip.open(filepath, 'rb') as sourcefp:
                        log.info("Verifying hash on vibid %s, payload %s, "
                                  "expected hash %s" %
                                 (vibid, payload.name, checksum.checksum))
                        hashedfp = HashedStream.HashedStream(sourcefp,
                                                             checksum.checksum,
                                                             "sha1")
                        inbytes = hashedfp.read(self.BUFFER_SIZE)
                        while inbytes:
                           inbytes = hashedfp.read(self.BUFFER_SIZE)
                        hashedfp.close()
                  else:
                     msg = "Payload '%s' of VIB %s at '%s' is missing" %\
                            (payload.name, vibid, filepath)
                     raise Errors.InstallationError(None, msg)
               except Exception as e:
                  msg = "Failed to verify checksum for payload %s: %s" % (payload.name, e)
                  log.error("%s" % msg)
                  raise Errors.ChecksumVerificationError(msg)
         if not checksumfound:
            msg = "Failed to find checksum for payload %s" % payload.name
            log.error("%s" % msg)
            raise Errors.ChecksumVerificationError(msg)


   def OpenPayloadFile(self, vibid, payload, read = True, write = False):
      """Creates and returns a File-like object for either reading from
         or writing to a given payload.  One of read or write must be True, but
         ready and write cannot both be true.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
            * read    - Set to True to get a File object for reading
                        from the payload.
            * write   - Set to True to get a File object for writing
                        to the payload.
         Returns:
            A File-like object, must support read (for read), write (for
            write), close methods.
            None if the desired read/write is not supported.
         Exceptions:
            AssertionError    - neither read nor write is True, or both are true
            InstallationError - Cannot open file to write
      """
      Installer.OpenPayloadFile(self, vibid, payload, read = read, write = write)
      if read == True:
         localfile = None
         if vibid in self.bootbank.db.profile.vibIDs:
            vibstate = self.bootbank.db.profile.vibstates[vibid]
            rootpath = self.bootbank.path
         elif self.bootbankstate & self.BOOTBANK_STATE_UPDATED and \
               vibid in self.altbootbank.db.profile.vibIDs:
            vibstate = self.altbootbank.db.profile.vibstates[vibid]
            rootpath = self.altbootbank.path
         elif self.bootbankstate & self.BOOTBANK_STATE_STAGED and \
               vibid in self.stagebootbank.db.profile.vibIDs:
            vibstate = self.stagebootbank.db.profile.vibstates[vibid]
            rootpath = self.stagebootbank.path
         else:
            msg = 'Cannot find VIB %s in BootbankInstaller' % vibid
            raise Errors.InstallationError([vibid], msg)

         if payload.name in vibstate.payloads:
            localfile = os.path.join(rootpath, vibstate.payloads[payload.name])
         else:
            msg = ("VIB %s found in BootbankInstaller (%s), but payload %s is "
                   "not found in the VIB" % (vibid, rootpath, payload.name))
            raise Errors.InstallationError([vibid], msg)

         try:
            return open(localfile, 'rb')
         except EnvironmentError as e:
            msg = 'Error in opening payload %s of VIB %s at %s for read: %s' % (
                  payload.name, vibid, localfile, e)
            raise Errors.InstallationError([vibid], msg)

      elif write == True:
         # bootbank installer only supports BOOT, VGZ and TGZ payload
         if payload.payloadtype not in self.SUPPORTED_PAYLOADS:
            log.debug("Payload %s of type '%s' in VIB '%s' is not supported by "
                      "BootbankInstaller." % (payload.payloadtype, payload.name,
                         vibid))
            return None

         bootbankfile = os.path.join(self.stagebootbank.path, payload.localname)
         try:
            fp = open(bootbankfile, 'wb')
         except EnvironmentError as e:
            self.Cleanup()
            msg = 'Can not open %s to write payload %s: %s' % (bootbankfile,
                  payload.name, e)
            raise Errors.InstallationError(None, msg)
         return fp

      return None

   def Cleanup(self):
      """Cleans up after a Transaction that has been started, perhaps from
         a previous instantiation of the Installer class.
         Unmount and remove stagebootbank.
      """
      self._CleanupStageBootbank()

   def CompleteStage(self):
      """Complete the staging bootbank by writing out the new imgdb
         and the new boot.cfg.

         Exceptions:
            InstallationError
      """
      imgprofile = self.stagebootbank.db.profile
      bootorder = imgprofile.GetBootOrder(self.SUPPORTED_PAYLOADS,
                                          self.SUPPORTED_VIBS)

      #TODO: need to accomodate custom boot option bewteen kernel and regular
      # module
      self.stagebootbank.bootcfg.build = imgprofile.GetEsxVersion()


      try:
         _, p = bootorder[0]
         self.stagebootbank.bootcfg.kernel = p.localname
         modules = [p.localname for (vibid, p) in bootorder[1:]]
         self.stagebootbank.bootcfg.modules = modules

         self.stagebootbank.bootcfg.UpdateState(Bootcfg.BOOTSTATE_STAGED)
         self.stagebootbank.updated = self.bootbank.updated + 1
         self.stagebootbank.Save()
      except Exception as e:
         self.Cleanup()
         msg = 'Failed to finish bootbank stage: %s'  % (e)
         raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

   def CacheNewImage(self, srcinstlr):
      log.info('Caching LiveImage to stagebootbank...')
      if not self.stagebootbank:
         self._SetupStageBootbank()
      self.stagebootbank.ClearContent()
      self.stagebootbank.bootcfg.UpdateState(Bootcfg.BOOTSTATE_EMPTY)
      self.stagebootbank.bootcfg.Save()

      profile = srcinstlr.database.profile.Copy()
      self.stagebootbank.db.vibs = profile.vibs
      self.stagebootbank.db.profiles.AddProfile(profile)

      # Read payloads from live image
      kernelpayloads = []
      for vibid in profile.vibIDs:
         for payload in srcinstlr.database.vibs[vibid].payloads:

            payload.localname = profile.vibstates[vibid].payloads[payload.name]

            # Get dest file obj, skip if payload is not supported
            dst = self.OpenPayloadFile(vibid, payload, write=True, read=False)
            if dst is None:
               continue

            log.debug("About to write payload '%s' of VIB %s to '%s'" % (payload.name,
               vibid, self.stagebootbank.path))

            src = None
            # Get src from bootbank
            if vibid in self.bootbank.db.profile.vibstates \
                  and  (payload.name in
                        self.bootbank.db.profile.vibstates[vibid].payloads):
               vibstate = self.bootbank.db.profile.vibstates[vibid]
               srcfile = os.path.join(self.bootbank.path,
                     vibstate.payloads[payload.name])
               log.debug('Using %s from bootbank as source' % (srcfile))
               src = open(srcfile, 'rb')
            # Get src from live image for newly installed module
            elif payload.payloadtype != payload.TYPE_BOOT:
               log.debug('Using source from %s' % (
                  srcinstlr.__class__.__name__))
               src = srcinstlr.OpenPayloadFile(vibid, payload, read=True,
                     write=False)
            else:
               log.error('Could not found payload %s from sources' % (
                  payload.name))

            if src == None and HostInfo.HostOSIsSimulator():
               dst.close()
               # delete the dest file
               bootbankfile = os.path.join(self.stagebootbank.path,
                  payload.localname)
               log.info("HostSimulator: Deleting unnecessary destination file %s."
                  % (bootbankfile))
               os.remove(bootbankfile)
               continue

            if src == None:
               msg = ('Error in opening payload %s of VIB %s from installer '
                      '%s' % (payload.name, vibid, srcinstlr.installertype))
               raise Errors.InstallationError([vibid,], msg)

            filesize = 0
            while True:
               inbytes = src.read(self.BUFFER_SIZE)
               if not inbytes:
                  break
               dst.write(inbytes)
               filesize += len(inbytes)
            log.debug('Total of %d bytes were written.' % (filesize))
            src.close()
            dst.close()

      self.CompleteStage()
      return self.Remediate(liveimage=True)

   def _UpdateBootloader(self, imgprofile):
      """Update the bootloader: For bootbank installs, during
         remediation check to see if the esx-base vib is being
         installed/upgraded and if the vib has the bootloader
         payload. If the payload is present in the vib, then
         this function mounts the payload tardisk and runs a
         script that updates the bootloader.
      """
      log.info('Checking for bootloader update')
      try:
         for vibid in imgprofile.vibstates:
            vib = self.altbootbank.db.vibs[vibid]
            if vib.name != 'esx-base':
               continue
            if 'btldr' not in imgprofile.vibstates[vibid].payloads:
               break
            #if the bootloader installer payload is found in
            #the base tardisk, run the internal script to modify
            #the bootloader.
            try:
               tmppath = '/tmp'
               filename = imgprofile.vibstates[vibid].payloads['btldr']
               srcpath = os.path.join(self.altbootbank.path, filename)
               dstpath = os.path.join(tmppath, filename)

               with open(srcpath, 'rb') as srcfp:
                  dstfp = EsxGzip.GunzipFile(dstpath, 'wb')

                  #copy the compressed payload to a temporary location
                  #in an uncompressed form so that it can be mounted.
                  inbytes = srcfp.read(self.BUFFER_SIZE)
                  dstfp.write(inbytes)
                  while inbytes:
                     inbytes = srcfp.read(self.BUFFER_SIZE)
                     dstfp.write(inbytes)
               dstfp.close()

               try:
                  #unmount any exisiting tardisk with this name.
                  #If unmount fails just overlay the tardisk with
                  #a differnt name.
                  os.remove('/tardisks/%s' % filename)
               except OSError as e:
                  log.warn('Failed unmounting bootloader tardisk %s: %s' % (filename, e))
                  filename = 'tmp-%s' % (filename)

               #mount tardisk
               if os.path.exists(SECURE_MOUNT_SCRIPT):
                  mountcmd = [SECURE_MOUNT_SCRIPT, vibid, "btldr", dstpath,
                              filename]
               else:
                  # for VUM single reboot, the host may not have securemount
                  mountcmd = 'mv %s /tardisks/%s' % (dstpath, filename)
               rc, out = runcommand(mountcmd)
               if rc != 0:
                  msg = ('Failed mounting bootloader tardisk %s: non-zero code returned'
                         '\nreturn code: %d\noutput: %s'
                         % (dstpath, rc, byteToStr(out)))
                  raise Exception(msg)

               #execute script
               execmd = '/usr/lib/vmware/bootloader-installer/install-bootloader'
               rc, out = runcommand(execmd)
               if rc != 0:
                  msg = ('Execution of command %s failed: non-zero code returned'
                         '\nreturn code: %d\noutput: %s'
                         % (execmd, rc, byteToStr(out)))
                  raise Exception(msg)

               #unmount the bootloader tardisk when temp filename is used
               if filename.startswith('tmp-'):
                  try:
                     os.remove('/tardisks/%s' % filename)
                  except OSError as e:
                     log.warn('Failed unmounting bootloader tardisk %s after'
                              ' the update was successful: %s' % (filename, e))

            except Exception as e:
               msg = 'Failed updating the bootloader: %s' % (e)
               raise Errors.InstallationError(vibid, msg)
      except Errors.InstallationError as e:
         raise
      except:
         msg = ('Unknown error: %s'
                '\nSkipping updating the bootloader' % (sys.exc_info()[0]))
         log.error(msg)

   def Remediate(self, checkmaintmode=False, liveimage=False):
      """For bootbank installs, the remediation consists of a reboot. Since
         the core engine does not carry out the reboot, we simply NOP.
         Returns:
            A Boolean, True if a reboot is needed.
         Exceptions:
            HostNotChanged    - If no staged ImageProfile
            InstallationError -
      """
      if not self.bootbankstate & self.BOOTBANK_STATE_STAGED:
         msg = "Bootbank is not yet staged, nothing to remediate."
         raise Errors.HostNotChanged(msg)

      adds, removes = self.stagedatabase.profile.Diff(self.database.profile)

      # Need to copy the staged image from stagebootbank to altbootbank.
      self.altbootbank.FileCopy(self.stagebootbank)

      self.altbootbank.Load(raiseerror=False)
      imgprofile = self.altbootbank.db.profile

      # TODO: Verify bootbank with respect to imgprofile
      self.altbootbank.Verify()

      self.altbootbank.bootcfg.modules.append(BootBank.DB_FILE)

      # If esx-base is in adds list, esxi version is updated.
      # This is useful for bootloader update and featurestate reset.
      esx_updated = False
      for vibid in adds:
         vib = self.altbootbank.db.vibs[vibid]
         if vib.name == 'esx-base':
            esx_updated = True
            break

      if os.path.isfile(BACKUP_SH):
         try:
            rc, out = runcommand('%s 0 %s' % (BACKUP_SH,
               self.altbootbank.path))
         except RunCommandError as e:
            msg = ('Failed to run %s: %s' % (BACKUP_SH, e))
            raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

         if rc != 0:
            msg = ('Error in running backup script: non-zero code returned.'
                   '\nreturn code: %d\noutput: %s' % (rc, byteToStr(out)))
            raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

         # When esxi is updated, we want to drop the features.gz written by
         # backup.sh, and let the profile's default features.gz take effect.
         # This will allow featurestate change to show up after reboot.
         if esx_updated:
            features_gz = 'features.gz'
            staged_features = os.path.join(self.stagebootbank.path, features_gz)
            altbootbank_features = os.path.join(self.altbootbank.path,
                                                features_gz)
            if os.path.exists(staged_features):
               log.debug('Copying feature config file %s to %s.' %
                         (staged_features, altbootbank_features))
               shutil.copy2(staged_features, altbootbank_features)
      else:
         log.debug("Skipping backup script '%s', which was not found" % (BACKUP_SH))

      statefile = os.path.join(self.altbootbank.path, 'state.tgz')
      if os.path.isfile(statefile):
         self.altbootbank.bootcfg.modules.append('state.tgz')

      log.info("boot config of '%s' is being updated, %d" %
            (self.altbootbank.path, self.altbootbank.updated))
      self.altbootbank.updated = self.bootbank.updated + 1
      if liveimage:
         self.altbootbank.bootcfg.UpdateState(Bootcfg.BOOTSTATE_SUCCESS,
                                              liveimage=True)
      else:
         self.altbootbank.bootcfg.UpdateState(Bootcfg.BOOTSTATE_UPDATED)

      # Keep kernel option from current bootbank a customer might have specified
      # The boot options stored in esx.conf are persisted by backup.sh script in
      # useropts.gz
      self.altbootbank.bootcfg.kernelopt = self.bootbank.bootcfg.kernelopt

      try:
         self.altbootbank.bootcfg.Save()
      except Exception as e:
         msg = 'Failed to remediate the host: %s' % (e)
         raise Errors.InstallationError(str(imgprofile.vibIDs), msg)

      # Check for bootloader update when esxi is updated
      if esx_updated:
         self._UpdateBootloader(imgprofile)

      return True

