#!/usr/bin/python
########################################################################
# Copyright (C) 2010-2018 VMWare, Inc.                                 #
# All Rights Reserved                                                  #
########################################################################

import fnmatch
import logging
import os
import re
import json
import shutil
import stat
import tarfile
import time
import socket

from vmware import runcommand
from vmware import vsi

from .. import Database
from .. import Errors
from .. import Vib
from .. import VibCollection
from ..Utils import EsxGzip
from ..Utils import HostInfo
from ..Utils import PathUtils
from ..Utils import Ramdisk
from ..Utils.Misc import byteToStr
from . import Installer

log = logging.getLogger('LiveImageInstaller')
LIVEINST_MARKER = 'var/run/update_altbootbank'

FAILURE_WARNING = '''It is not safe to continue. Please reboot the host \
immediately to discard the unfinished update.'''

SECURE_MOUNT_SCRIPT = '/usr/lib/vmware/secureboot/bin/secureMount.py'

#
# If an image is live staged, we don't need to stage it to bootbank again
#
class LiveImage(object):
   '''Encapsulates attributes of LiveImage
      Attributes:
         * root             - root of the LiveImage, default '/'
         * stageroot        - root of the staging directory
         * stagedatadir     - Temporary directory for payloads files (unziped)
         * database         - An instance of Database for live image
         * stagedatabase    - An instance of Database.Database representing the
                              package database for staged image. None if not
                              yet staged.
         * isstaged         - Boolean, indicate whether there is a staged image
   '''
   DB_DIR = os.path.join('var', 'db', 'esximg')
   STAGE_DIR = 'tmp/stageliveimage'
   STAGE_RAMDISK_NAME = 'stageliveimage'

   CHKCONFIG_DB = 'etc/chkconfig.db'
   SERVICES_DIR = 'etc/init.d'
   TARDISK_BACKUP_DIR = '/tmp/tardiskbackup'
   TARDISK_BACKUP_RAMDISK_NAME = 'tardiskbackup'
   STATE_BACKUP_DIR = 'var/run/statebackup'

   IsMockup = False
   mockupFile = '/etc/vmware/hostd/mockup-host-config.xml'
   if os.path.isfile(mockupFile):
      IsMockup = True

   if IsMockup:
      profileFilePath = '/etc/profile'
      pattern = "echo In container"
      with open(profileFilePath, 'r') as profileFile:
         for line in profileFile:
            m = re.search(pattern, line)
            if m:
               ctId = line.strip().split(' ')[-1]
               TARDISK_BACKUP_RAMDISK_NAME = ctId + '-' + \
                                             TARDISK_BACKUP_RAMDISK_NAME
               break

   def __init__(self, root = '/', stagedir = None):
      self._root = root
      self.database = Database.Database(os.path.join(self._root, self.DB_DIR),
            dbcreate = True)
      if stagedir is None:
         stagedir = self.STAGE_DIR
      self._stageroot = os.path.join(self._root, stagedir)
      self._stagedata = os.path.join(self._stageroot, 'data')
      self._stagedbpath = os.path.join(self._stageroot, 'imgdb')
      self._stageindicator = os.path.join(self._stageroot, 'staged')
      self._stagedb = None
      self._tardisksdir = os.path.join(root, 'tardisks')
      self._chkconfigdb = os.path.join(root, self.CHKCONFIG_DB)
      self._servicesdir = os.path.join(root, self.SERVICES_DIR)
      self._statebackupdir = os.path.join(root, self.STATE_BACKUP_DIR)
      self._servicestates = {"enabled": set(), "disabled": set(),
                             "added": set(), "removed": set(),
                             "upgradevibs": set()}
      self._triggers = [FirewallRefreshTrigger(),
                        HostServiceRefreshTrigger(),
                        WBEMServiceEnableTrigger()]

      # Dictionary holding information for live vib failure recovery
      self._recovery = {"stoppedservices": [],
                        "startedservices": [],
                        "unmountedtardisks": [],
                        "mountedtardisks": [],
                        "jumpstartplugins": set(),
                        "rcscripts": set(),
                        "removedconfigs": []}

      # List holding config information for preserving config during
      # live vib update
      self._backupconfigs = []

      self.Load()

   @property
   def root(self):
      return self._root

   @property
   def stageroot(self):
      return self._stageroot

   @property
   def stagedatadir(self):
      return self._stagedata

   @property
   def stagedatabase(self):
      if self.isstaged:
         return self._stagedb
      return None
   @property
   def tardisksdir(self):
      return self._tardisksdir

   @property
   def chkconfigdb(self):
      return self._chkconfigdb

   @property
   def isstaged(self):
      return os.path.isfile(self._stageindicator)

   def Load(self):
      '''Load LiveImage and staging info.

         Exceptions:
            * DatabaseFormatError
            * DatabaseIOError
      '''
      self.database.Load()
      if self.isstaged:
         try:
            self._stagedb = Database.Database(self._stagedbpath, dbcreate=False)
            self._stagedb.Load()
         except Exception as e:
            msg = ('Found a copy of staged image, but unable to load the database: '
                   '%s. Resetting live image as non-staged' % (e))
            log.warn(msg)
            self._UnSetStaged()
      # profile.vibs should contain Vib instance as Transaction only replies on
      # image profile to get detailed VIB info.
      if self.database.profile is not None:
         self.database.profile.vibs = self.database.vibs
      if self.stagedatabase and self.stagedatabase.profile is not None:
         self.stagedatabase.profile.vibs = self.stagedatabase.vibs

   def StartTransaction(self, imgprofile, imgsize):
      '''Reset staging directory to empty.
      '''
      problems = self._CheckHost(imgprofile)
      if problems:
         msg = 'Not enough resources to run the transaction: %s' % (problems)
         raise Errors.InstallationError([], msg)

      self._Setup(imgsize)
      self._stagedb = Database.Database(self._stagedbpath, addprofile=False)
      self._stagedb.vibs = imgprofile.vibs
      self._stagedb.profiles.AddProfile(imgprofile)

   def CompleteStage(self):
      '''Save staging database and staged indicator
      '''
      self._stagedb.Save()
      try:
         self._SetStaged()
      except Exception as e:
         msg = 'Failed to change the status of the staged image: %s' % (e)
         raise Errors.InstallationError([], msg)

   def _MaintenanceModeVibs(self, tgtvibs, adds, removes,
         checkmaintmode=True):
      '''Create list of vibs which require maintenance mode to finish the
         transaction.
         Parameters:
            * tgtvibs - VibCollection instance for the desired image profile
                        vibs
            * adds    - List of Vib IDs which will be added during the
                        transaction
            * removes - List of Vib IDs which will be removed the during the
                        transaction
         Returns: A tuple (mmoderemove, mmodeinstall); mmoderemove
            is a set of VIB IDs which requires maintenance mode to be removed;
            mmodeinstall is a set of VIB IDs which requires maintenance mode to
            be installed.
      '''
      mmoderemove = set()
      mmodeinstall = set()
      for vibid in adds:
         if tgtvibs[vibid].maintenancemode.install:
            mmodeinstall.update([vibid,])
      for vibid in removes:
         if self.database.vibs[vibid].maintenancemode.remove:
            mmoderemove.update([vibid,])
      return (mmoderemove, mmodeinstall)

   def Remediate(self, checkmaintmode=True):
      '''Live remove VIBs and enable new VIBs
      '''
      if self.stagedatabase.profile is None:
         msg = ('There is an error in the DB of staging directory: %s, Please '
                'try the update command again.' % (self._stagedbpath))
         raise Errors.InstallationError([], msg)

      adds, removes = self.stagedatabase.profile.Diff(self.database.profile)
      vibstoadd = [self.stagedatabase.vibs[vibid] for vibid in adds]
      vibstoremove = [self.database.vibs[vibid] for vibid in removes]

      self._UpdateServiceStates(vibstoadd, vibstoremove)
      self._UpdateTriggers(vibstoadd, vibstoremove)

      # check maintenance mode required for the transaction
      mmoderemove, mmodeinstall = self._MaintenanceModeVibs(
            self.stagedatabase.vibs, adds, removes)
      if mmoderemove or mmodeinstall:
         msg = ('MaintenanceMode is required to '
                'remove: [%s]; install: [%s].'
                   % (', '.join(mmoderemove),
                      ', '.join(mmodeinstall)))
         log.debug(msg)
         if checkmaintmode:
            mmode = HostInfo.GetMaintenanceMode()
            if not mmode:
               raise Errors.MaintenanceModeError(msg)
         else:
            log.warning('MaintenanceMode check was skipped...')

      # save config files
      cmd = '/sbin/backup.sh 0'
      try:
         RunCmdWithMsg(cmd, raiseexception=False)
      except Exception as e:
         log.warning('There was an error to run [%s]: %s' % (cmd, e))

      # XXX Verify staged image

      # Backup state.tgz from /bootbank or /altbootbank (by checking
      # LIVEINST_MARKER) to /var/run/statebackup/ for live vib failure handling
      try:
         if not os.path.exists(self._statebackupdir):
            os.makedirs(self._statebackupdir)
         if os.path.isfile(os.path.join(self.root, LIVEINST_MARKER)):
            src = '/altbootbank/state.tgz'
         else:
            src = '/bootbank/state.tgz'
         dest = os.path.join(self._statebackupdir, 'state.tgz')
         shutil.copy2(src, dest)
      except Exception as e:
         log.warn("Error copying %s to %s: %s" % (src, dest, str(e)))

      # For vib removal case, we need to check the modified config file by
      # the existence of .# file. No need to check the original file permission
      # since it could be changed.
      vibstoadd_name = set([v.name for v in vibstoadd])
      for vib in vibstoremove:
         if vib.name in vibstoadd_name:
            for fn in vib.GetModifiedConf():
               self._backupconfigs.append(fn)

      try:
         self._RemoveVibs(removes)
         self._AddVibs(adds)

         # Run secpolicytools, PR 581102
         # 1. setup policy rules for vmkaccess (/etc/vmware/secpolicy/*)
         # 2. Apply the security label for any new tardisks that have been
         #    installed and mounted.
         cmd = '/sbin/secpolicytools -p'
         try:
            RunCmdWithMsg(cmd, raiseexception=False)
         except Exception as e:
            log.warning('There was an error to run [%s]: %s' % (cmd, e))

         # Start daemons in order
         self._StartServices()

         # Invoke post inst triggers
         for trigger in self._triggers:
            trigger.Run()
      except Errors.InstallationError as e:
         self._HandleLiveVibFailure(e.msg)
      except runcommand.RunCommandError as e:
         msg = "%s\n%s" % (e, FAILURE_WARNING)
         self._HandleLiveVibFailure(msg)
      finally:
         # Remove the tardiskbackup ramdisk
         Ramdisk.RemoveRamdisk(self.TARDISK_BACKUP_RAMDISK_NAME,
                               self.TARDISK_BACKUP_DIR)
         # Delete state.tgz backup
         try:
            os.remove(os.path.join(self._statebackupdir, 'state.tgz'))
         except Exception as e:
            log.warning("Error deleting backup state.tgz: %s" % str(e))

      self._UpdateDB()

      # mark system live installed
      markerfile = os.path.join(self.root, LIVEINST_MARKER)
      msg = ("There was an error creating flag file '%s',"
             "configuration changes after installation might be lost "
             "after reboot." % (markerfile))
      try:
         open(markerfile, 'w').close()
      except IOError as e:
         log.warning("Creating file failed: %s" % e)
         log.warning(msg)


   def Cleanup(self):
      '''Remove data in staging area.
      '''
      # this is a bit tricky, have to track init, tardisk mapping,
      try:
         if os.path.exists(self._stageroot):
            # Unset the stage indicator first to avoid the race condition on
            # it with vib.get, vib.list or profile.get command running
            # at the same time.
            self._UnSetStaged()
            # remove staging ramdisk
            Ramdisk.RemoveRamdisk(self.STAGE_RAMDISK_NAME, self._stageroot)
      except EnvironmentError as e:
         msg = 'Cannot remove live image staging directory: %s' % (e)
         raise Errors.InstallationError([], msg)
      self._stagedb = None

   def _RemoveVibs(self, removes):
      log.debug('Starting to live remove VIBs: %s' % (', '.join(removes)))

      shutdownscripts = set()
      for vibid in removes:
         vib = self.database.vibs[vibid]
         log.info('Live removing %s-%s' % (vib.name, vib.version))
         # run shutdown script
         scripts = self._GetFilesFromVib(vib,
               lambda x: fnmatch.fnmatch(x, "etc/vmware/shutdown/shutdown.d/*"))
         shutdownscripts.update(scripts)

      self._ShutdownServices()

      # Module unloading
      log.debug('Starting to run etc/vmware/shutdown/shutdown.d/*')
      for script in shutdownscripts:
         RunCmdWithMsg(script)

      # Calculate the size of unmounted tardisks
      # and create a new ramdisk for unmounted tardisk
      totalsize = 0
      for vibid in removes:
         for name, localname in \
               list(self.database.profile.vibstates[vibid].payloads.items()):
            totalsize = totalsize + \
                  os.path.getsize(os.path.join(self._tardisksdir, localname))

      totalsize = int(round(totalsize/1024/1024)) + 1
      Ramdisk.CreateRamdisk(totalsize, self.TARDISK_BACKUP_RAMDISK_NAME, \
                            self.TARDISK_BACKUP_DIR)

      # umount tardisks
      for vibid in removes:
         vib = self.database.vibs[vibid]

         # Collect modified config for recovery
         for fn in vib.GetModifiedConf():
            self._recovery['removedconfigs'].append(fn)

         for name, localname in \
               list(self.database.profile.vibstates[vibid].payloads.items()):
            log.debug('Trying to unmount payload [%s] of VIB %s' % (name, vibid))
            # Make a copy of the tardisk to be unmounted for recovery
            src = os.path.join(self._tardisksdir, localname)
            dest = os.path.join(self.TARDISK_BACKUP_DIR, localname)
            log.debug('Copying tardisk from %s to %s' % (src, dest))
            try:
               shutil.copy2(src, dest)
            except IOError:
               log.warning('Unable to copy file from %s to %s' % (src, dest))
            self._UnmountTardisk(localname)
            self._recovery["unmountedtardisks"].append([vibid, name, localname])

         # Collect jumpstart plugins and rc scripts for recovery
         plugins = self._GetFilesFromVib(vib,
               lambda x: fnmatch.fnmatch(x, "usr/libexec/jumnpstart/plugins/*"))
         for fn in plugins:
            self._recovery["jumpstartplugins"].add(os.path.basename(fn))

         scripts = self._GetFilesFromVib(vib,
               lambda x: fnmatch.fnmatch(x, "etc/rc.local.d/*"))
         self._recovery["rcscripts"].update(scripts)

      log.debug('done')

   def _AddVibs(self, adds):
      log.debug('Starting to enable VIBs: %s' % (', '.join(adds)))

      jumpstartplugins = set()
      rcscripts = set()

      # ordering non-overlay and overlay VIB, so non-overlay VIBs being mounted
      # first
      regvibs = []
      overlayvibs = []
      for vibid in adds:
         vib = self.stagedatabase.vibs[vibid]
         if vib.overlay:
            overlayvibs.append(vibid)
         else:
            regvibs.append(vibid)

      for vibid in regvibs + overlayvibs:
         vib = self.stagedatabase.vibs[vibid]

         plugins = self._GetFilesFromVib(vib,
               lambda x: fnmatch.fnmatch(x, "usr/libexec/jumpstart/plugins/*"))
         for fn in plugins:
            jumpstartplugins.add(os.path.basename(fn))

         scripts = self._GetFilesFromVib(vib,
               lambda x: fnmatch.fnmatch(x, "etc/rc.local.d/*"))
         rcscripts.update(scripts)

         log.debug('Live installing %s-%s' % (vib.name, vib.version))

         for name, localname in \
            list(self.stagedatabase.profile.vibstates[vibid].payloads.items()):
            log.debug('Trying to mount payload [%s]' % (name))
            self._MountTardiskSecure(vibid, name, localname)
            self._recovery["mountedtardisks"].append(localname)

         #TODO: Resource group management

         # Restore config from /var/run/statebackup/state.tgz before
         # running jumpstart and rc script. Need to do following checks
         # before restoring config files:
         # 1. config files are in self._backupconfigs.
         # 2. same files are added when installing new vib.
         # 3. sticky bit are on for the added files.
         try:
            statetgz = tarfile.open(
                   os.path.join(self._statebackupdir, 'state.tgz'), 'r:gz')
            localtgz = tarfile.open(fileobj=statetgz.extractfile('local.tgz'))
            for fn in vib.filelist:
               normalfile = PathUtils.CustomNormPath('/' + fn)
               if os.path.isfile(normalfile) and \
                    os.stat(normalfile).st_mode & stat.S_ISVTX and \
                    fn in self._backupconfigs:
                  log.info("Extracting %s from state.tgz" % fn)
                  localtgz.extract(fn, '/')
            localtgz.close()
            statetgz.close()
         except Exception as e:
            log.warning("Error restoring config files for vib %s: %s" % \
                                                        (vib.name, str(e)))

      if jumpstartplugins:
         self._StartJumpStartPlugins(jumpstartplugins)

      if rcscripts:
         self._RunRcScripts(rcscripts)

      log.debug('Enabling VIBs is done.')

   def _UpdateDB(self):
      self.database.vibs = self.stagedatabase.vibs
      self.database.profiles = self.stagedatabase.profiles
      self.database.Save()

   def _UpdateVib(self, newvib):
      """Update missing properties of vib metadata

         Parameters:
            * newvib   - The new vib to use as source
         Returns:
            None if the update succeeds, Exception otherwise
         Exceptions:
            VibMetadataError
      """
      try:
         self._stagedb.vibs[newvib.id].SetSignature(newvib.GetSignature())
         self._stagedb.vibs[newvib.id].SetOrigDescriptor(newvib.GetOrigDescriptor())
      except Exception as e:
         raise Errors.VibMetadataError(e)

   def _Setup(self, imgsize):
      self.Cleanup()
      try:
         # Live image use uncompressed payload, while installation size is
         # compress size.
         # Exact ramdisk size is hard to figure, we assume compression ratio
         # will be at most 4, RAM will be consumed as needed under this max
         ramdisk_max_size = int((imgsize / 1024 / 1024) * 4)
         Ramdisk.CreateRamdisk(ramdisk_max_size, self.STAGE_RAMDISK_NAME,
                               self._stageroot)
         os.makedirs(self._stagedata)
      except Exception as e:
         msg = 'There was a problem setting up staging area: %s' % (e)
         raise Errors.InstallationError([], msg)

   def _SetStaged(self):
      if not os.path.isfile(self._stageindicator):
         f = open(self._stageindicator, 'w')
         f.close()

   def _UnSetStaged(self):
      if os.path.exists(self._stageindicator):
         os.unlink(self._stageindicator)

   def _CheckHost(self, adds):
      '''TODO: Check the host environment to make sure:
            * enough memory to expand VIB
            * enough userworlds memory
      '''
      return []

   def _MountTardisk(self, tardisk):
      tardiskpath = os.path.join(self._stagedata, tardisk)
      cmd = 'mv %s /tardisks/' %(tardiskpath)
      # Sometimes the filesystem will report that the file is busy
      # or in use.  This is usually a transient error and simply retrying
      # might fix the issue.
      RunCmdWithRetries(cmd, 'Mounting %s...' % (tardisk))

   def _MountTardiskSecure(self, vibid, payload, tardisk):
      tardiskpath = os.path.join(self._stagedata, tardisk)
      args = [SECURE_MOUNT_SCRIPT, vibid, payload, tardiskpath]
      RunCmdWithMsg(args, 'Mounting %s...' % (tardisk))

   def _UnmountTardisk(self, tardisk):
      cmd = 'rm /tardisks/%s' % (tardisk)
      # Sometimes the filesystem will report that the file is busy
      # or in use.  This is usually a transient error and simply retrying
      # might fix the issue.
      RunCmdWithRetries(cmd, 'Unmounting %s...' % (tardisk))

   @staticmethod
   def _GetFilesFromVib(vib, matchfn):
      files = []
      for fn in vib.filelist:
         normfile = PathUtils.CustomNormPath(fn)
         if matchfn(normfile):
            # return script with absolute path so the script can be run from any
            # directory.
            files.append(os.path.join('/', normfile.strip()))
      return files

   @staticmethod
   def _GetUpgradeVibs(adds, removes):
      addvibs = VibCollection.VibCollection()
      for vib in adds:
         addvibs.AddVib(vib)
      removevibs = VibCollection.VibCollection()
      for vib in removes:
         removevibs.AddVib(vib)

      allvibs = VibCollection.VibCollection()
      allvibs += addvibs
      allvibs += removevibs

      upgradevibs = set()
      scanner = allvibs.Scan()
      for vibs in (addvibs, removevibs):
         upgradevibs.update(scanner.GetUpdatesSet(vibs))
         upgradevibs.update(scanner.GetDowngradesSet(vibs))

      return upgradevibs

   def _UpdateServiceStates(self, adds, removes):
      enabled = set()
      disabled = set()
      serviceadds = set()
      serviceremoves = set()

      # set of vibids involving upgrade/downgrade
      upgradevibs = self._GetUpgradeVibs(adds, removes)

      filterfn = lambda x: fnmatch.fnmatch(x, "etc/init.d/*")
      for vib in adds:
         files = self._GetFilesFromVib(vib, filterfn)
         serviceadds.update(files)
         if vib.id in upgradevibs:
            self._servicestates["upgradevibs"].update(files)
      for vib in removes:
         files = self._GetFilesFromVib(vib, filterfn)
         serviceremoves.update(files)
         if vib.id in upgradevibs:
            self._servicestates["upgradevibs"].update(files)

      upgrades = serviceadds & serviceremoves
      serviceadds -= upgrades
      serviceremoves -= upgrades

      if upgrades:
         cmd = ["/sbin/chkconfig", "-B", self._chkconfigdb, "-D",
               self._servicesdir, "-l"]
         rc, out = runcommand.runcommand(cmd)
         out = byteToStr(out)
         if rc:
            msg = "Failed to get chkconfig service list: %s" % out
            raise Errors.InstallationError([], msg)

         for line in out.splitlines():
            line = line.strip()
            if line.endswith(" on"):
               servicepath = os.path.join(self._servicesdir, line[:-3].strip())
               enabled.add(servicepath)
            elif line.endswith(" off"):
               servicepath = os.path.join(self._servicesdir, line[:-4].strip())
               disabled.add(servicepath)

         for script in upgrades:
            if script in enabled:
               self._servicestates["enabled"].add(script)
            elif script in disabled:
               self._servicestates["disabled"].add(script)
      self._servicestates["added"].update(serviceadds)
      self._servicestates["removed"].update(serviceremoves)

   def _GetEnabledServices(self):
      cmd = ["/sbin/chkconfig", "-B", self._chkconfigdb, "-D",
             self._servicesdir, "-i", "-o"]
      rc, out = runcommand.runcommand(cmd)
      out = byteToStr(out)
      if rc:
         msg = "Failed to obtain chkconfig service list: %s" % out
         raise Errors.InstallationError([], msg)
      return [line.strip() for line in out.splitlines()]

   def _ShutdownServices(self):
      '''Shutdown services according to the order in chkconfig.db. It is assumed
         that all the services are managed by chkconfig.
      '''
      tostop = self._servicestates["removed"] | self._servicestates["enabled"]
      for service in reversed(self._GetEnabledServices()):
         cmd = [service, 'stop']
         if service in self._servicestates["upgradevibs"]:
            cmd.append("upgrade")
         else:
            cmd.append("remove")
         if service in tostop:
            RunCmdWithMsg(cmd)
            self._recovery["stoppedservices"].append(cmd)

   def _SetServiceEnabled(self, service, enabled):
      service = os.path.basename(service)
      cmd = ["/sbin/chkconfig", "-B", self._chkconfigdb, "-D",
             self._servicesdir, service, (enabled and "on" or "off")]
      rc, _ = runcommand.runcommand(cmd)
      if rc:
         action = enabled and "enable" or "disable"
         msg = "Unable to %s service %s\n%s" % (action, service,
                                                FAILURE_WARNING)
         raise Errors.InstallationError([], msg)

   def _StartServices(self):
      # Need to enable/disable any services which were previously explicitly
      # enabled/disabled.
      enabled = set(self._GetEnabledServices())
      for service in enabled & self._servicestates["disabled"]:
         self._SetServiceEnabled(service, False)

      for service in self._servicestates["enabled"] - enabled:
         self._SetServiceEnabled(service, True)

      tostart = self._servicestates["enabled"] | self._servicestates["added"]
      for service in self._GetEnabledServices():
         if service in tostart:
            cmd = [service, "start"]
            if service in self._servicestates["upgradevibs"]:
               cmd.append("upgrade")
            else:
               cmd.append("install")

            RunCmdWithMsg(cmd)
            self._recovery["startedservices"].append(cmd)

   def _StartJumpStartPlugins(self, plugins):
      cmd = ["/sbin/jumpstart", "--method=start", "--invoke",
             "--plugin=%s" % ",".join(plugins)]
      rc, out = runcommand.runcommand(cmd)
      if rc:
         log.warn("Error loading one or more JumpStart plugins: %s" %
                  byteToStr(out))

   def _RunRcScripts(self, scripts):
      for script in scripts:
         if os.stat(script).st_mode & stat.S_IXUSR:
            RunCmdWithMsg(script)
         else:
            log.warn("Script: %s is not executable. Skipping it." % script)

   def _UpdateTriggers(self, adds, removes):
      '''Activate trigger for VIB installation or removal
      '''
      for vib in adds:
         for trigger in self._triggers:
            trigger.Match(vib, 'add')
      for vib in removes:
         for trigger in self._triggers:
            trigger.Match(vib, 'remove')

   def _HandleLiveVibFailure(self, msg):
      '''Handle live vib installation failure by:
            * Shutdown the services from new vibs
            * Unmount the tardisks from new vibs
            * Mount the unmounted tardisks
            * Restore configuration
            * Start jumpstart and rc scripts from removed vibs
            * Start the services from removed vibs
            * Invoke triggers
            * Raise LiveInstallationError
      '''
      log.warn("Handling Live Vib Failure: %s" % msg)

      # Shutdown the services from new vibs
      for (service,_,_) in reversed(self._recovery["startedservices"]):
         try:
            cmd = [service, 'stop']
            if service in self._servicestates["upgradevibs"]:
               cmd.append("upgrade")
            else:
               cmd.append("remove")
            RunCmdWithMsg(cmd, raiseexception=False)
         except Exception as e:
            log.warn("Unable to shutdown service %s: %s" % (cmd, str(e)))

      # Unmount the tardisks from new vibs
      for localname in reversed(self._recovery["mountedtardisks"]):
         try:
            self._UnmountTardisk(localname)
         except Exception as e:
            log.warn("Unable to unmount tardisk %s: %s" % (localname, str(e)))

      # Mount the unmounted tardisks
      for vibid, name, localname in reversed(self._recovery["unmountedtardisks"]):
         try:
            src = os.path.join(self.TARDISK_BACKUP_DIR, localname)
            dest = os.path.join(self._stagedata, localname)
            shutil.move(src, dest)
            self._MountTardiskSecure(vibid, name, localname)
         except Exception as e:
            log.warn("Unable to mount tardisk %s: %s" % (localname, str(e)))

      # Restore the removed configs from /var/run/statebackup/state.tgz
      try:
         statetgz = tarfile.open(
                    os.path.join(self._statebackupdir, 'state.tgz'), 'r:gz')
         localtgz = tarfile.open(fileobj=statetgz.extractfile('local.tgz'))
         for fn in self._recovery["removedconfigs"]:
            log.info('Extracting %s from state.tgz' %fn)
            localtgz.extract(fn, '/')
         localtgz.close()
         statetgz.close()
      except Exception as e:
         log.warn("Error restoring removed configs: %s" % str(e))

      # Start jumpstart plugins and rc scripts
      try:
         if self._recovery["jumpstartplugins"]:
            self._StartJumpStartPlugins(self._recovery["jumpstartplugins"])
         if self._recovery["rcscripts"]:
            self._RunRcScripts(self._recovery["rcscripts"])
      except Exception as e:
         log.warn("Error %s" % str(e))

      # Start the stopped services
      for (service,_,_) in reversed(self._recovery["stoppedservices"]):
         try:
            cmd = [service, 'start']
            if service in self._servicestates["upgradevibs"]:
               cmd.append("upgrade")
            else:
               cmd.append("install")
            RunCmdWithMsg(cmd, raiseexception=False)
         except Exception as e:
            log.warn("Unable to start service %s: %s" % (cmd, str(e)))

      # Invoke triggers
      for trigger in self._triggers:
         try:
            trigger.Run()
         except Exception as e:
            log.warn("Error running %s: %s" % (trigger, str(e)))

      # Raise LiveInstallationError
      raise Errors.LiveInstallationError([], msg)

class LiveImageInstaller(Installer):
   '''LiveImageInstaller is the Installer class to live install/remove VIBs for
      live system.

      Attributes:
         * database - A Database.Database instance of the live system
   '''
   installertype = "live"
   priority = 5
   SUPPORTED_VIBS = set([Vib.BaseVib.TYPE_BOOTBANK,])
   SUPPORTED_PAYLOADS = set([Vib.Payload.TYPE_TGZ, Vib.Payload.TYPE_VGZ])
   BUFFER_SIZE = 8*1024

   def __init__(self, root = '/'):
      self.liveimage = LiveImage(root)
      self.problems = list()

   @property
   def database(self):
      return self.liveimage.database

   @property
   def stagedatabase(self):
      return self.liveimage.stagedatabase

   def StartTransaction(self, imgprofile, imgstate = None, preparedest = True,
                        forcebootbank = False, **kwargs):
      """Initiates a new installation transaction. Calculate what actions
         need to be taken.

         This method only works on staging directory

         Parameters:
            * imgprofile  - The ImageProfile instance representing the
                            target set of VIBs for the new image
            * imgstate    - The state of current HostImage, one of IMGSTATE_*
            * preparedest - Boolean, if True, then prepare the destination.
                            Set to false for a "dry run", to avoid changing
                            the destination.
            * forcebootbank - Boolean, if True, skip install of live image
                              even if its eligible for live install
         Returns:
            A tuple (installs, removes, staged), installs and removes are list
            of VIB IDs for HostImage.Stage() to install to the destination and
            to remove from the destination, in order to make it compliant
            with imgprofile. If LiveImage has already staged the imgprofile,
            staged is True.
            If there is nothing to do, (None, None, False) is returned.
         Exceptions:
            InstallationError
      """
      # Skip if reboot required VIBs have been installed
      from ..HostImage import HostImage
      if forcebootbank:
         msg = 'Nothing to do for live install - live installation has been ' \
               'disabled.'
         self.problems.append(msg)
         log.debug(msg)
         return (None, None, False)

      if imgstate not in (HostImage.IMGSTATE_FRESH_BOOT,
            HostImage.IMGSTATE_LIVE_UPDATED):
         msg = ('Only reboot-required installations are possible right now as '
                'a reboot-required installation has been done.')
         self.problems.append(msg)
         log.debug(msg)
         return (None, None, False)

      if self.database.profile is None:
         msg = 'No ImageProfile found for live image'
         raise Errors.InstallationError(msg)

      imgprofile = imgprofile.Copy()
      unsupported = imgprofile.vibIDs - self.GetSupportedVibs(imgprofile.vibs)
      for vibid in unsupported:
         imgprofile.RemoveVib(vibid)

      staged = False
      if self.stagedatabase is not None:
         if self.stagedatabase.profile and \
               self.stagedatabase.profile.vibIDs == imgprofile.vibIDs:
            staged = True
         else:
            self.liveimage.Cleanup()

      adds, removes = imgprofile.Diff(self.database.profile)

      if staged:
         return (adds, removes, staged)

      problems = self._CheckTransaction(imgprofile, adds, removes)
      if problems:
         log.debug('The transaction is not supported for live install:\n%s' % (problems))
         self.problems = problems
         return (None, None, False)

      imgsize = self.GetInstallationSize(imgprofile)

      if preparedest and (removes or adds):
         self.liveimage.StartTransaction(imgprofile, imgsize)

      return (adds, removes, staged)

   def VerifyPayloadChecksum(self, vibid, payload):
      """Verify the checksum of a given payload.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
         Returns:
            None if verification succeeds, Exception otherwise
         Exceptions:
            ChecksumVerificationError
            InstallationError
      """
      if payload.payloadtype in self.SUPPORTED_PAYLOADS:
         # TODO: we are currently only checking the SUPPORTED_PAYLOADS
         # for this installer. Once we have the ability to lookup
         # modules hashes from the kernel, we should enable checking
         # for other boot modules.
         checksumfound = False
         for checksum in payload.checksums:
            if checksum.checksumtype == "sha-1":
               checksumfound = True
               try:
                  if vibid in self.database.profile.vibstates:
                     vibstate = self.database.profile.vibstates[vibid]
                  else:
                     msg = 'Could not locate VIB %s in LiveImageInstaller' % (vibid)
                     raise Errors.InstallationError(None, msg)
                  if payload.name in vibstate.payloads:
                     tardiskname = vibstate.payloads[payload.name]
                  else:
                     msg = "Payload name '%s' of VIB %s not in LiveImage DB" %\
                           (payload.name, vibid)
                     raise Errors.InstallationError(None, msg)
                  tardiskchecksum = vsi.get('/system/visorfs/tardisks/%s/sha1hash' % tardiskname)
                  tardiskchecksum = ''.join(("%02x" % x) for x in tardiskchecksum)
                  if tardiskchecksum != checksum.checksum:
                     msg = ("Payload checksum mismatch. "
                            "Expected <%s>, kernel loaded <%s>" % (checksum.checksum, tardiskchecksum))
                     raise Exception(msg)
               except Exception as e:
                  msg = "Failed to verify checksum for payload %s: %s" % (payload.name, e)
                  log.error("%s" % msg)
                  raise Errors.ChecksumVerificationError(msg)
         if not checksumfound:
            msg = "Failed to find checksum for payload %s" % payload.name
            log.error("%s" % msg)
            raise Errors.ChecksumVerificationError(msg)

   def UpdateVibDatabase(self, newvib):
      """Update missing properties of vib metadata
         New vibs are always installed in the liveimage

         Parameters:
            * newvib   - The new vib to use as source
         Returns:
            None if the update succeeds, Exception otherwise
         Exceptions:
            VibMetadataError
      """
      self.liveimage._UpdateVib(newvib)

   def OpenPayloadFile(self, vibid, payload, read = True, write = False):
      """Creates and returns a File-like object for either reading from
         or writing to a given payload.  One of read or write must be True, but
         ready and write cannot both be true.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
            * read    - Set to True to get a File object for reading
                        from the payload.
            * write   - Set to True to get a File object for writing
                        to the payload.
         Returns:
            A File-like object, must support read (for read), write (for
            write), close methods.
            None if the desired read/write is not supported.
         Exceptions:
            AssertionError    - neither read nor write is True, or both are true
            InstallationError - Cannot open file to write or read
      """
      Installer.OpenPayloadFile(self, vibid, payload, read, write)


      if read:
         if vibid not in self.database.profile.vibstates:
            msg = 'Could not locate VIB %s in LiveImageInstaller' % (vibid)
            raise Errors.InstallationError(None, msg)

         vibstate = self.database.profile.vibstates[vibid]
         if payload.name not in vibstate.payloads:
            msg = "Payload name '%s' of VIB %s not in LiveImage DB" % (
                  payload.name, vibid)
            raise Errors.InstallationError(None, msg)

         if (payload.payloadtype in self.SUPPORTED_PAYLOADS or \
             payload.payloadtype == Vib.Payload.TYPE_BOOT):

            filepath = os.path.join(self.liveimage.tardisksdir,
                  vibstate.payloads[payload.name])
            if os.path.isfile(filepath):
               #NOTE: No name and time info in the gzip stream
               return EsxGzip.GunzipFile(filepath, 'rb')
            else:
               msg = "Payload '%s' of VIB %s at '%s' is missing" % (
                     payload.name, vibid, filepath)
               if HostInfo.HostOSIsSimulator():
                  log.info("HostSimulator: %s" % (msg))
                  return None
               else:
                  raise Errors.InstallationError(None, msg)
         else:
            return None
      else:
         if payload.payloadtype not in self.SUPPORTED_PAYLOADS:
            log.debug("Payload %s of type '%s' in VIB '%s' is not supported by "
                      "LiveImageInstaller." % (payload.name, payload.payloadtype,
                         vibid))
            return None

         filepath = os.path.join(self.liveimage.stagedatadir, payload.localname)
         try:
            fp = EsxGzip.GunzipFile(filepath, 'wb')
         except EnvironmentError as e:
            msg = 'Can not open %s to write payload %s: %s' % (filepath,
                  payload.name, e)
            raise Errors.InstallationError([], msg)
         return fp

      return None

   def Cleanup(self):
      '''Cleans up the live image staging area.
      '''
      try:
         self.liveimage.Cleanup()
      except Exception:
         pass

   def CompleteStage(self):
      """Complete the staging of live image by writing out the database and
         staged indicator to staging directory.

         Exceptions:
            InstallationError
      """
      self.liveimage.CompleteStage()

   def Remediate(self, checkmaintmode=True):
      """Live remove and install VIB payloads

         For each VIB to remove, shutdown scripts will be executed and payloads
         will be unmounted. For each VIB to install, payloads will be mounted
         and init scripts will be executed.

         Returns:
            A Boolean, always False, as a reboot is not needed.
         Exceptions:
            InstallationError -
            HostNotChanged    - If there is no staged ImageProfile
      """
      if self.liveimage.isstaged:
         self.liveimage.Remediate(checkmaintmode)
      else:
         msg = 'LiveImage is not yet staged, nothing to remediate.'
         raise Errors.HostNotChanged(msg)

      return False

   def _CheckTransaction(self, imageprofile, adds, removes):
      '''Check the transaction to see if there are any logical reasons that
         prevent live installation of the transaction.
            * For VIBs to be removed: require liveremoveok
            * For VIBs to be installed: require liveinstallok
            * No VIBs to be installed are overlaid by existing VIB
            * No VIBs to be removed are overlaid by existing VIB
      '''
      problems = []

      # Build a map from file path to the file states. File state is a bit map
      # to indicate which group the file belongs to.
      filestates = {}
      keepreg, keepoverlay, removereg, removeoverlay, addreg, addoverlay = \
         1, 2, 4, 8, 16, 32

      keeps = self.database.profile.vibIDs - set(removes)
      groups = ((keeps, self.database.vibs, (keepreg, keepoverlay)),
                (removes, self.database.vibs, (removereg, removeoverlay)),
                (adds, imageprofile.vibs, (addreg, addoverlay)))
      for vibids, vibs, flags in groups:
         for vibid in vibids:
            vib = vibs[vibid]
            for filepath in vib.filelist:
               if filepath == '' or filepath.endswith('/'):
                     continue

               filepath = PathUtils.CustomNormPath('/' + filepath)
               ind = vib.overlay and 1 or 0
               if filepath in filestates:
                  filestates[filepath] |= flags[ind]
               else:
                  filestates[filepath] = flags[ind]

      unsupported = {
         keepoverlay | removereg  : \
            "File to be removed is overlaid by existing VIB",
         keepoverlay | removereg | addreg : \
            "File to be removed/installed is overlaid by existing VIB",
         keepoverlay | addreg : \
            "File to be installed is overlaid by existing VIB",
         #XXX: visorFS to support resurrect?
         keepreg | removeoverlay : \
            "File to be removed overlays existing VIB",
         keepreg | removeoverlay | addoverlay : \
            "File to be removed/installed overlays existing VIB",
            }

      for filepath in filestates:
         if filestates[filepath] in unsupported:
            problem = "%s : %s" % (unsupported[filestates[filepath]], filepath)
            problems.append(problem)

      for vibid in adds:
         # liveinstallok must be True for new VIB
         if not imageprofile.vibs[vibid].liveinstallok:
            problem = 'VIB %s cannot be live installed.' % (vibid)
            problems.append(problem)

      for vibid in removes:
         # liveremoveok must be Ture for VIB to be removed
         if not self.database.vibs[vibid].liveremoveok:
            problem = 'VIB %s cannot be removed live.' % (vibid)
            problems.append(problem)

      return problems

   def GetSupportedVibTypes(self):
      # Support locker VIB for PXE host so live install/remove
      # will keep tools VIB untouched in the image profile.
      if HostInfo.IsPxeBooting():
         return self.SUPPORTED_VIBS | set([Vib.BaseVib.TYPE_LOCKER,])
      else:
         return self.SUPPORTED_VIBS

class PostInstTrigger(object):
   '''Base class for system wide post installation trigger action. It does not
      implement any action.

      Attributes:
         * NAME - The name of the trigger
         * matchingvibs - List of VIB objects which trigger the class action
   '''
   NAME = ''

   def __init__(self):
      self.matchingvibs = []

   def Match(self, vib, operation):
      '''Check the properties of a VIB instance against the operation (add/remove)
         to be performed. Add the VIB instance to matchingvibs list if install/remove
         this vib will need to run trigger action.

         Parameters:
            * vib - A VIB instanace, which will be installed or removed.
      '''
      raise NotImplementedError('Must instantiate a subclass of PostInstTrigger')

   def Run(self):
      '''Fire trigger action. This should be invoked after VIBs have been live
         installed/removed.

         NOTE: sub-class needs to implement _run() mehtod or override Run
               method.
      '''
      if self.matchingvibs:
         log.info("Executing post inst trigger : '%s'" % (self.NAME))
         self._run()

class Sfcb():
   '''Handles communications with sfcbd when vib is added or removed by
      using a unix domain socket to send control characters. Now sfcbd
      may be administratively up or down so must first get app config
      to know app state.'''
   def __init__(self, operation):
      '''operation is add/remove for vib, client is either sfcb or wsman
         that needs updating its runtime state when vib is added or removed'''
      cmd = 'localcli --formatter json system wbem get'
      rpt = json.loads(RunCmdWithMsg(cmd))
      self.isRunning = rpt['Enabled']
      self.wsmanRunning = rpt['WS-Management Service']
      self.operation = operation

   def StartWbemServices(self):
      if self.isRunning:
         return
      cmd = 'localcli system wbem set --enable=true'
      RunCmdWithMsg(cmd)
      self.isRunning = True

   def SendUpdateRequest(self):
      '''This routine creates a UDS pipe to sfcbd and sends control code.'''
      sck = None
      try:
         sck = self._Connect()
         # see cayman_sfcb.git control.c for control codes
         UPDATE_CMD = 'A'
         if self.wsmanRunning:
            UPDATE_CMD += 'W'
         self._SendCmd(sck, UPDATE_CMD)
      finally:
         if sck:
            self._CloseSock(sck)

   def _Connect(self):
      '''return socket to sfcb mgmt interface  '''
      ctrl_path = '/var/run/sfcb.ctl'
      sck = socket.socket(socket.AF_UNIX, socket.SOCK_STREAM)
      retries = 3
      while retries > 0:
         retries -= 1
         try:
            sck.connect(ctrl_path)
            return sck
         except Exception as err:
            time.sleep(3)
            log.error("Connect to '%s' err: %s" % (ctrl_path, err))
            if retries == 0:
               self._CloseSock(sck)
               raise

   def _CloseSock(self, sck):
      try:
         sck.shutdown(1) # close transmit, done sending
         sck.close()
      except Exception:
         pass

   def _SendCmd(self, sck, cmd):
      '''write all bytes in cmd to sck else throw exception'''
      totalsent = 0
      while totalsent < len(cmd):
         sent = sck.send(cmd[totalsent:].encode('utf-8'))
         if sent == 0:
            log.error("Unable to transmit command %s" % cmd)
            raise Exception("Unable to transmit command %s" % cmd)
         totalsent = totalsent + sent

class FirewallRefreshTrigger(PostInstTrigger):
   '''FirewallRefreshTrigger is the trigger to refresh firewall settings
   '''
   NAME = 'Firewall Refresh Trigger'

   def Match(self, vib, operation):
      if LiveImage._GetFilesFromVib(vib,
            lambda x: fnmatch.fnmatch(x, "etc/vmware/firewall/*")):
         self.matchingvibs.append(vib)
         log.info('%s is required to install/remove %s' % (self.NAME, vib.id))

   def _run(self):
      cmds = ['export VI_USERNAME=vpxuser; /sbin/esxcli network firewall refresh',
              '/bin/vim-cmd -U vpxuser hostsvc/refresh_firewall']
      for cmd in cmds:
         try:
            RunCmdWithMsg(cmd, raiseexception=False)
         except Exception as e:
            log.warning('There was an error to run [%s]: %s' % (cmd, e))

class HostServiceRefreshTrigger(PostInstTrigger):
   '''HostServiceRefreshTrigger is the trigger to refresh host service system
   '''
   NAME = 'Service System Refresh Trigger'

   def Match(self, vib, operation):
      if LiveImage._GetFilesFromVib(vib,
            lambda x: fnmatch.fnmatch(x, "etc/vmware/service/*")):
         self.matchingvibs.append(vib)
         log.info('%s is required to install/remove %s' % (self.NAME, vib.id))

   def _run(self):
      cmd = '/bin/vim-cmd -U vpxuser hostsvc/refresh_service'
      try:
         RunCmdWithMsg(cmd, raiseexception=False)
      except Exception as e:
         log.warning('There was an error to run [%s]: %s' % (cmd, e))

class WBEMServiceEnableTrigger(PostInstTrigger):
   '''WBEMServiceEnableTrigger is the trigger to enable wbem services when
      custom cim provider is installed or if running reload CIM providers.
      Trigger does different things depending on configured state of sfcbd.
      see: esxcli system wbem get
      If sfcbd is running, send control character(s) to signal to sfcbd
      to reconfigure after add/remove of a cim provider and to restart openwsmand
      if configured to run. If sfcbd is not running and a cim provider is installed
      this trigger will enable wbem services/start up sfcbd/openwsmand.
   '''
   NAME = 'WBEM Service Enable Trigger'

   def Match(self, vib, operation):
      self.operation = operation
      self.vibRemove = True if operation == 'remove' else False
      provider_path = 'var/lib/sfcb/registration/*-providerRegister'
      if LiveImage._GetFilesFromVib(vib,
            lambda x: fnmatch.fnmatch(x, provider_path)):
          self.matchingvibs.append(vib)

   def _run(self):
      log.debug("Processing '%s' to request sfcb to reload providers" % \
                   (self.NAME))
      try:
         sfcb = Sfcb(self.operation)
         if self.operation == 'add' and not sfcb.isRunning:
             sfcb.StartWbemServices()
             return
         sfcb.SendUpdateRequest()
      except Exception as err:
         log.warning("Unable to reconfigure SFCB: %s" % err)

def RunCmdWithMsg(cmd, title='', raiseexception=True):
   if title:
      log.debug(title)
   else:
      log.debug('Running [%s]...' % (cmd))
   rc, out = runcommand.runcommand(cmd)
   out = byteToStr(out)
   if out:
      log.debug("output: %s" % (out))

   if rc != 0:
      msg = ('Error in running %s:\nReturn code: %s'
             '\nOutput: %s\n%s' % (cmd, rc, out, FAILURE_WARNING))
      if raiseexception:
         raise Errors.InstallationError([], msg)
      else:
         log.warning(msg)
   return out

def RunCmdWithRetries(cmd, msg, numRetries = 3):
   for i in range(1, numRetries+1):
      try:
         out = RunCmdWithMsg(cmd, msg)
         return out
      except Exception as e:
         # raise when the last try fails
         if i == numRetries:
            raise Errors.InstallationError([], str(e))
         log.info("Received error: %s\nTrying again. Attempt #%d", e, i)
         time.sleep(i)
   return None
