#!/usr/bin/python
########################################################################
# Copyright (C) 2010 VMWare, Inc.                                      #
# All Rights Reserved                                                  #
########################################################################
import logging
import os
import shutil
import subprocess
import time

from vmware import runcommand
from .. import Database
from .. import Errors
from .. import Vib
from . import Installer
from ..Utils import HostInfo

LOCKER_ROOT = '/locker/packages/'

log = logging.getLogger('LockerInstaller')

class UntarFile(object):
   '''UntarFile class provides write and close methods to untar a tar.gz stream
      to a directory. close before EOF will raise InstallationError.
   '''
   def __init__(self, dst):
      '''Class constructor.
         Parameters:
            * dst - the directory path to which .tar.gz stream to be extracted.
      '''
      self._dst = dst
      self._cmd = '/bin/tar xzf - -C %s' % (self._dst)
      self._p = subprocess.Popen(self._cmd, shell=True, stdin=subprocess.PIPE,
            stderr=subprocess.PIPE)

   def write(self, data):
      self._p.stdin.write(data)

   def close(self, timeout=30):
      '''Close untar stream.
         Parameters:
            * timeout - timeout before any attempt to terminate worker
                        subprocess.
      '''
      self._p.stdin.close()
      err = self._p.stderr.read()
      rc = None
      # Give untar process a bit time to finish
      length = 0.2
      while timeout > 0:
         rc = self._p.poll()
         if rc == 0:
            break
         time.sleep(length)
         timeout -= length
         length *= 2

      if rc is None:
         runcommand.terminateprocess(self._p)

      if self._p.returncode != 0:
         msg = ('Error in closing untar process, return code: %s\n'
                'stderr: %s'% (rc, err))
         raise Errors.InstallationError(None, msg)

class LockerInstaller(Installer):
   '''LockerInstaller is the Installer class for Locker package files.
      LockerInstaller is only supported on regular booted host. Tools package on
      PXE host is handled by LiveImageInstaller.
      Attributes:
         * database - A Database.Database instance of product locker
         * stagedatabase - Always None, there is no staging support.
   '''
   installertype = 'locker'
   DB_DIR = os.path.join('var', 'db', 'locker')
   priority = 20

   SUPPORTED_VIBS = set([Vib.BaseVib.TYPE_LOCKER, ])
   SUPPORTED_PAYLOADS = set(['tgz', ])

   def __init__(self, root=None):
      if root is None:
         self._root = LOCKER_ROOT
      else:
         self._root = root

      # For PXE host, tools payload is handled by LiveImageInstaller only
      if HostInfo.IsPxeBooting():
         raise Errors.InstallerNotAppropriate('Host booted from PXE server or'
               ' there was an error to get boot type. LockerInstaller is not'
               ' supported')
      self.database = Database.Database(os.path.join(self._root, self.DB_DIR),
            dbcreate = True, addprofile=False)
      self.Load()
      self._updateindicator = os.path.join(self._root, 'lockerupdated')
      self.problems = list()

   @property
   def stagedatabase(self):
      return None

   @property
   def root(self):
      return self._root

   def Load(self):
      '''Load locker database.
      '''
      try:
         self.database.Load()
      except Exception as e:
         log.warning('Locker DB is not available: %s' % (e))

   def StartTransaction(self, imgprofile, imgstate = None, preparedest = True,
                        forcebootbank = False, **kwargs):
      """Initiates a new installation transaction. Calculate what actions
         need to be taken.

         This method will change product locker

         Parameters:
            * imgprofile  - The ImageProfile instance representing the
                            target set of VIBs for the new image
            * imgstate    - The state of current HostImage, one of IMGSTATE_*
            * preparedest - Boolean, if True, then prepare the destination.
                            Set to false for a "dry run", to avoid changing
                            the destination.
            * forcebootbank - Boolean, if True, skip install of live image
                              even if its eligible for live install
            * stageonly     - If True, do nothing as there is enough space to
                              stage.
         Returns:
            A tuple (installs, removes, staged), installs and removes are list
            of VIB IDs for HostImage.Stage() to install to the destination and
            to remove from the destination, in order to make it compliant
            with imgprofile.
            If there is nothing to do, (None, None, False) is returned.
         Exceptions:
            * InstallationError
      """
      stageonly = kwargs.get('stageonly', False)
      if stageonly:
         msg = 'Stage only is not supported for LockerInstaller.'

         self.problems.append(msg)
         return (None, None, False)

      supported = self.GetSupportedVibs(imgprofile.vibs)

      keeps = set(self.database.vibs.keys()) & supported
      removes = set(self.database.vibs.keys()) - keeps
      adds = supported - keeps

      if preparedest and (removes or adds):
         self._RemoveVibs(self.database.vibs, removes)

         self.database.Clear()
         self._UnSetUpdated()
         for vibid in supported:
            self.database.vibs.AddVib(imgprofile.vibs[vibid])

      return (adds, removes, False)

   def OpenPayloadFile(self, vibid, payload, read = False, write = True):
      '''Creates and returns a File-like object for writing to a given payload.
         Only write is supported.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
            * read    - Must be False; ready is not supported
            * write   - Set to True to get a File object for writing
                        to the payload.
         Returns:
            A File-like object, must support write and close methods.
            None if the desired read/write is not supported.
         Exceptions:
            AssertionError    - neither read nor write is True, or both are true
            InstallationError - Cannot open file to write or read
      '''
      Installer.OpenPayloadFile(self, vibid, payload, read, write)
      if write == True:
         if payload.payloadtype not in self.SUPPORTED_PAYLOADS:
            log.debug("Payload %s of type '%s'  in VIB '%s' is not supported by "
                  "LockerInstaller." % (payload.name, payload.payloadtype,
                         vibid))
            return None
         return UntarFile(self._root)
      else:
         raise ValueError('OpenPayloadFile for read is not supported in '
               'LockerInstaller')

   def Cleanup(self):
      '''Clean up locker packages directory. Since there is no space for
         staging, locker packages content will be cleaned.
      '''
      self.database.Clear()
      try:
         shutil.rmtree(self.database.dbpath)
         shutil.rmtree(self._root)
         os.makedirs(self._root)
      except Exception as e:
         log.warning('There was an error in cleaning up product locker: %s'
               % (e))

   def CompleteStage(self):
      '''Complete the staging of live image by writing out the database.
      '''
      self.database.Save()
      self._SetUpdated()

   def Remediate(self, checkmaintmode=True):
      """Nothing to do here, as there is no space to stage in Locker.

         Returns:
            A Boolean, always False, as a reboot is not needed.
         Exceptions:
            * HostNotChanged - If host is not changed in previous Stage command.
      """
      if os.path.exists(self._updateindicator):
         self._UnSetUpdated()
         return False
      else:
         raise Errors.HostNotChanged('Locker files not chaged.')

   def _SetUpdated(self):
      if not os.path.isfile(self._updateindicator):
         f = open(self._updateindicator, 'w')
         f.close()

   def _UnSetUpdated(self):
      if os.path.exists(self._updateindicator):
         os.unlink(self._updateindicator)

   def _RemoveVibs(self, allvibs, reomves):
      for vibid in reomves:
         log.debug('Removing files from productLocker for VIB %s' % (vibid))
         for filepath in allvibs[vibid].filelist:
            filepath = filepath.lstrip('/')
            realpath = os.path.join(self._root, filepath)
            if os.path.isfile(realpath):
               try:
                  os.unlink(realpath)
               except EnvironmentError as e:
                  log.warning('Unable to delete file [%s]: %s, skipping...' %
                        (realpath, e))

