#!/usr/bin/python
########################################################################
# Copyright (C) 2010 - 2015 VMWare, Inc.                                      #
# All Rights Reserved                                                  #
########################################################################

class Installer(object):
   """Installer is the base class behind all the *Installer classes
      and serves as an interface.  It does not implement any real
      installation functionality.

      Attributes:
         * database - A Database instance representing the package
                      database associated with the installer
   """
   SUPPORTED_VIBS = set()
   SUPPORTED_PAYLOADS = set()

   def __init__(self):
      """Constructor for this Installer class.
         Should determine if this class of installer should be created
         within the runtime environment.
         If so, initialize any databases.
         Exceptions:
            InstallerNotAppropriate - Environment is not appropriate for
               this type of installer.
      """
      raise NotImplementedError("Must instantiate a subclass of Installer")

   def __del__(self):
      """Destructor. Should always clean up after installations.
      """

   def StartTransaction(self, imgprofile, **kwargs):
      """Initiates a new installation transaction. Calculate what actions
         need to be taken.  Prepare installation destination as necessary.
         This method may change the installation destination.

         Parameters:
            * imgprofile - The ImageProfile instance representing the
                           target set of VIBs for the new image
            * preparedest - Boolean, if True, then prepare the destination.
                           Set to false for a "dry run", to avoid changing
                           the destination.
            * forcebootbank - Boolean, if True, skip install of live image
                           even if its eligible for live install
            * imgstate   - One of the HostImage.IMGSTATE constants
         Returns:
            A tuple (installs, removes), each of which is a list of VIB
            IDs for HostImage.Stage() to install to the destination and
            to remove from the destination, in order to make it compliant
            with imgprofile.
            If there is nothing to do, (None, None) is returned.
         Exceptions:
            InstallationError
      """
      raise NotImplementedError("Must instantiate a subclass of Installer")

   def OpenPayloadFile(self, vibid, payload, read = True, write = False):
      """Creates and returns a File-like object for either reading from
         or writing to a given payload.  One of read or write must be True.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
            * read    - Set to True to get a File object for reading
                        from the payload.
            * write   - Set to True to get a File object for writing
                        to the payload.
         Returns:
            A File-like object, must support read (for read), write (for
            write), close methods.
            None if the desired read/write is not supported.
         Exceptions:
            AssertionError - neither read nor write is True.
            InstallationError
      """
      assert (read or write) and not (read and write), \
         "Only one of read or write can and must be True"

   def VerifyPayloadChecksum(self, vibid, payload):
      """Verify the checksum of a given payload.

         Parameters:
            * vibid   - The Vib id containing the payload
            * payload - The Vib.Payload instance to read or write
         Returns:
            None if verification succeeds, Exception otherwise
         Exceptions:
            ChecksumVerificationError
            InstallationError
      """

   def UpdateVibDatabase(self, newvib):
      """Update missing properties of vib metadata

         Parameters:
            * newvib   - The new vib to use as source
         Returns:
            None if the update succeeds, Exception otherwise
         Exceptions:
            VibMetadataError
      """

   def Cleanup(self):
      """Cleans up after a Transaction that has been started, perhaps from
         a previous instantiation of the Installer class.
      """
      raise NotImplementedError("Must instantiate a subclass of Installer")

   def CompleteStage(self):
      """Do what is needed to complete the stage operation.

         Exceptions:
            InstallationError
      """
      raise NotImplementedError("Must instantiate a subclass of Installer")

   def Remediate(self):
      """Carry out the remediation operation.

         Returns:
            A Boolean, True if a reboot is needed.
         Exceptions:
            InstallationError
      """
      raise NotImplementedError("Must instantiate a subclass of Installer")

   def GetSupportedVibTypes(self):
      '''Return a set of supported vibtypes by the installer.
      '''
      return self.SUPPORTED_VIBS

   def GetSupportedVibs(self, vibs):
      '''Return a set of VIB IDs which are supported by the installer.
         Parameters:
            * vibs - A VibCollection instance.
      '''
      return set(vib.id for vib in vibs.values() if vib.vibtype in
            self.GetSupportedVibTypes())

   def GetInstallationSize(self, imgprofile):
      '''Return total size of payloads by supported by this installer.
         Parameter:
            An imageprofile
         Returns:
            Total byte size of the payloads supported by this installer
      '''
      totalsize = 0
      for vibid in imgprofile.vibIDs:
         vib = imgprofile.vibs[vibid]
         if vib.vibtype in self.SUPPORTED_VIBS:
            for payload in vib.payloads:
               if payload.payloadtype in self.SUPPORTED_PAYLOADS:
                  totalsize += payload.size
      return totalsize

