#!/usr/bin/python
########################################################################
# Copyright (C) 2010 VMWare, Inc.                                      #
# All Rights Reserved                                                  #
########################################################################
import hashlib
import logging
import os
import tempfile
import sys
import zipfile

if sys.version_info[0] >= 3:
   from urllib.request import url2pathname
   from urllib.parse import urlparse
else:
   from urllib import url2pathname
   from urlparse import urlparse

from . import Downloader
from . import Errors
from . import DepotCollection
from . import Vib

from .Utils import PathUtils

"Read, write, and extract esximage offline bundle zip files."

log = logging.getLogger('OfflineBundle')

HASH_BUF_SIZE = 16 * 1024

class OfflineBundle(object):
   """Class representing an esximage offline bundle zip, with methods to scan,
      extract, and write an offline bundle zip to a file."""
   DEPOT_INDEX = 'index.xml'

   def __init__(self, bundleurl):
      """Create a new OfflineBundle instance.
         Parameters:
            * bundleurl - Either a path to an offline bundle or the full remote
                          or local URL of the depot index.xml file. Bundle file
                          name must end with '.zip'.
      """
      self._bundleurl = bundleurl
      self._dc = DepotCollection.DepotCollection()

   def Load(self):
      ''' Read Depot metadata nodes. This is actually handled by
          DepotCollection.ConnectDepots method, but exception will be raised.

          Exceptions:
            BundleIOError - error reading from offline bundle or a depot
      '''
      try:
         self._dc.ConnectDepots([self._bundleurl,], ignoreerror=False)
      except Downloader.DownloaderError as e:
         msg = 'Error in downloading files: %s' % (e)
         raise Errors.BundleIOError(self._bundleurl, msg)

   @property
   def channels(self):
      return self._dc.channels

   @property
   def vibs(self):
      return self._dc.vibs

   @property
   def profiles(self):
      return self._dc.profiles

   @property
   def vibscandata(self):
      return self._dc.vibscandata

   def ScanVibs(self):
      self._dc.ScanVibs()

   def WriteBundleZip(self, dest, checkacceptance=True):
      '''Write bundle zip.
         Parameters:
            * dest            - A file path to write to.
            * checkacceptance - If True (the default), the acceptance level of
                                VIBs are validated as they are added to the
                                bundle zip.
         Exceptions:
            * BundleIOError      - Error in writing bundle zip file.
            * BundleFormatError  - If a depot metadata node or VIB is not under
                                   depot root directory.
            * VibSignatureError  - If acceptancecheck is true and acceptance
                                   level signature validation fails.
            * VibValidationError - If acceptancecheck is true and acceptance
                                   level XML schema validation fails.
      '''
      assert len(self._dc.depots) == 1, 'Only one depot is allowed'
      depotnode = self._dc.depots[0]

      try:
         bundle = zipfile.ZipFile(dest, 'w', zipfile.ZIP_DEFLATED)
      except EnvironmentError as e:
         msg = 'Error in opening file: %s' % (e)
         raise Errors.BundleIOError(dest, msg)

      depotroot = PathUtils.UrlDirname(depotnode.absurl)
      try:
         depotindex = depotnode.ToString()
         bundle.writestr(OfflineBundle.DEPOT_INDEX, depotindex)

         notificationfile = os.path.dirname(self._bundleurl)+"/notifications.zip"
         if os.path.exists(notificationfile):
            bundle.write(notificationfile,"notifications.zip")

         for vendornode in depotnode.children:
            self._AddNodeToBundle(bundle, depotroot, vendornode)
            for metanode in vendornode.children:
               self._AddNodeToBundle(bundle, depotroot, metanode, download=True)
         for vib in self._dc.vibs.values():
            self._AddVibToBundle(bundle, depotroot, vib, checkacceptance)
         bundle.close()
      except EnvironmentError as e:
         bundle.close()
         os.unlink(dest)
         msg = 'Error in writing bundle %s: %s' % (dest, e)
         raise Errors.BundleIOError(dest, msg)
      except Exception:
         bundle.close()
         os.unlink(dest)
         raise

   @staticmethod
   def _AddNodeToBundle(bundle, depotroot, node, download=False):
      log.debug('Adding DepotNode [%s] from %s' % (node.META_NODE_TAG,
         node.absurl))
      if node.absurl.startswith(depotroot):
         if download:
            fh, fn = tempfile.mkstemp()
            os.close(fh)
            try:
               d = Downloader.Downloader(node.absurl, local=fn)
               localfile = d.Get()
               bundle.write(localfile, node.absurl[len(depotroot):])
            finally:
               OfflineBundle._ForceRemoveFile(fn)
         else:
            bundle.writestr(node.absurl[len(depotroot):], node.ToString())
      else:
         msg = ("Node '%s' doesn't share the same root with the depot %s" % (
            node.absurl, depotroot))
         raise Errors.BundleFormatError(bundle.filename, msg)

   @staticmethod
   def _CheckPayloadDigest(vibid, payload, fobj):
      checksum = payload.GetPreferredChecksum()
      hashalgo = checksum.checksumtype.replace("-", "")
      try:
         hasher = hashlib.new(hashalgo)
         data = fobj.read(HASH_BUF_SIZE)
         while data:
            hasher.update(data)
            data = fobj.read(HASH_BUF_SIZE)
         digest = hasher.hexdigest().lower()
      except (ValueError, IOError) as e:
         msg = "Error validating payload digest: %s" % e
         raise Errors.VibPayloadDigestError(vibid, payload.name, msg)
      expected = checksum.checksum.lower()
      if digest != expected:
         msg = ("Calculated digest does not match expected result: "
                "%s calculated, %s expected." % (digest, expected))
         raise Errors.VibPayloadDigestError(vibid, payload.name, msg)

   @staticmethod
   def _AddVibToBundle(bundle, depotroot, vib, checkacceptance=True):
         log.debug('Adding VIB %s to bundle' % (vib.id))
         vurl = None
         for url in vib.remotelocations:
            if url.startswith(depotroot):
               vurl = url
               break

         if vurl is None:
            msg = 'Unable to locate %s under depot %s' % (vib.id, depotroot)
            raise Errors.BundleFormatError(bundle.filename, msg)

         scheme, netloc, path = urlparse(vurl)[:3]
         downloaded = False
         localfile = None
         if scheme == 'file':
            localfile = url2pathname(path)
         else:
            try:
               fh, fn = tempfile.mkstemp()
               os.close(fh)
               d = Downloader.Downloader(vurl, local=fn)
               localfile = d.Get()
               downloaded = True
            except Downloader.DownloaderError as e:
               log.info('Unable to download from %s, error [%s]. Trying next url...'
                     % (url, str(e)))
               OfflineBundle._ForceRemoveFile(fn)

         if localfile is None:
            msg = ("Unable to get VIB %s from any of the URLs %s" %
                  (vib.id, ', '.join(vib.remotelocations)))
            raise Errors.VibDownloadError(vib, msg)

         vibobj = None
         try:
            vibobj = Vib.ArFileVib.FromFile(localfile)
            if checkacceptance:
               vibobj.VerifyAcceptanceLevel()
            for payload, fobj in vibobj.IterPayloads():
               OfflineBundle._CheckPayloadDigest(vibobj.id, payload, fobj)
            try:
               bundle.write(localfile, vurl[len(depotroot):])
            except EnvironmentError as e:
               msg = 'Error adding VIB %s to bundle: %s' % (vib.name, e)
               raise Errors.BundleIOError(bundle.filename, msg)
         finally:
            if vibobj:
               vibobj.Close()
            if downloaded and localfile is not None:
               OfflineBundle._ForceRemoveFile(localfile)

   @staticmethod
   def _ForceRemoveFile(fn):
      if os.path.isfile(fn):
         try:
            os.unlink(fn)
         except EnvironmentError as e:
            log.info('Unable to clean up temp file %s: %s' % (fn, e))

if __name__ == '__main__':
   logging.basicConfig(level=logging.DEBUG)

   metaurl = sys.argv[1]
   dest = sys.argv[2]
   ob = OfflineBundle(metaurl)
   ob.Load()
   ob.WriteBundleZip(dest)
