#!/usr/bin/python
########################################################################
# Copyright (C) 2010-2018 VMWare, Inc.                                 #
# All Rights Reserved                                                  #
########################################################################

"High-level operations for installing and inventorying VIBs."

import tempfile
import logging
import os
import sys

if sys.version_info[0] >= 3:
   from urllib.parse import urlparse
else:
   from urlparse import urlparse

from . import Errors
from . import Database
from . import DepotCollection
from . import Downloader
from . import HostImage
from . import Vib
from . import VibCollection
from . import ImageProfile
from . import Metadata
from .Utils import PathUtils, Ramdisk, HostInfo
from .Version import Version

log = logging.getLogger("Transaction")

# The string that gets appended to an image profile name when its modified
UPDATEDPROFILESTR = "(Updated) "

# Patch the patcher feature switch
try:
   import featureState
   featureState.init(enableLogging=False)
   PATCH_PATCHER = featureState.EsxcliPatchPatcher
except Exception:
   PATCH_PATCHER = False

class InstallResult(object):
   """Holds the results of installation methods.
      Attributes:
         * installed  - A list of VIB IDs for the VIBs that were installed.
         * removed    - A list of VIB IDs for the VIBs that were removed.
         * skipped    - A list of VIB IDs for VIBs that were skipped and not
                        installed.  One reason is if keeponlyupdates=True was
                        passed to InstallVibsFromSources().
   """
   def __init__(self, **kwargs):
      self.installed = kwargs.pop('installed', list())
      self.removed   = kwargs.pop('removed', list())
      self.skipped   = kwargs.pop('skipped', list())
      self.exitstate = kwargs.pop('exitstate', Errors.NormalExit())


class Transaction(object):
   """Class with methods to install, remove, and inventory VIBs, and to scan a
      host or image against one or more depots."""

   def __init__(self):
      self.host = HostImage.HostImage()
      self.vibs = VibCollection.VibCollection()
      self.profiles = ImageProfile.ImageProfileCollection()

   def DownloadMetadatas(self, metadataUrls):
      '''Downloads metadatas to temporary files, parses them and
         builds up collections of vibs and image profiles.
         Raises:
            MetadataDownloadError
      '''
      Downloader.Downloader.setEsxupdateFirewallRule('true')
      try:
         self.vibs = VibCollection.VibCollection()
         self.profiles = ImageProfile.ImageProfileCollection()
         for metaUrl in metadataUrls:
            f = tempfile.NamedTemporaryFile()
            try:
               d = Downloader.Downloader(metaUrl, f.name)
               mfile = d.Get()
            except Downloader.DownloaderError as e:
               raise Errors.MetadataDownloadError(metaUrl, None, str(e))

            m = Metadata.Metadata()
            m.ReadMetadataZip(mfile)
            m.SetVibRemoteLocations(d.url)
            f.close()    # This deletes the temp file
            self.vibs += m.vibs
            self.profiles += m.profiles
      finally:
         Downloader.Downloader.setEsxupdateFirewallRule('false')

   def ParseDepots(self, depotUrls):
      '''Parse depot or offline bundle to build collections of vibs and image
         profiles.
         Paramaters:
            * depotUrls - A list of URLs to depots or offline bundle zip file.
                          For offline bundle URL, only server local file path is
                          supported. Offline bundle must end with '.zip'.
         Raises:
            ValueError  - If offline bundle is not of server local file path.
      '''
      remotebundles = []
      for url in depotUrls:
         if url.lower().endswith('.zip') and not PathUtils.IsDatastorePath(url):
            scheme = urlparse(url)[0]
            if scheme not in ('file', ''):
               remotebundles.append(url)

      if remotebundles:
         msg = ('Only server local file path is supported for offline '
                'bundles. (%s) seem to be remote URIs.' % (', '.join(
                     remotebundles)))
         raise ValueError(msg)

      dc = DepotCollection.DepotCollection()
      dc.ConnectDepots(depotUrls, ignoreerror=False)
      self.vibs += dc.vibs
      self.profiles += dc.profiles

   def GetProfile(self):
      '''Gets an image profile from the host with the vibs VibCollection filled
         out.
         Parameters: None
         Raises:
            InstallationError if a profile is not present.
      '''
      curprofile = self.host.GetProfile()
      if curprofile is None or len(curprofile.vibIDs) == 0:
         msg = ('No image profile is found on the host or '
                'image profile is empty. An image profile is required to '
                'install or remove VIBs.  To install an image profile, '
                'use the esxcli image profile install command.')
         raise Errors.InstallationError(None, msg)

      curprofile.vibs = self.host.GetInventory()
      if len(curprofile.vibs) == 0:
         msg = 'No VIB database was found on the host'
         raise Errors.InstallationError(None, msg)

      curprofile.acceptancelevel = self.host.GetHostAcceptance()
      return curprofile

   def GetVibsFromSources(self, vibUrls, metaUrls, vibspecs,
                          fillfrommeta=True, depotUrls=[], newestonly=True):
      '''Get VIBs instance from either vibUrls or metadata. VIBs from both
         URLs and metadata are merged.
         Parameters: See InstallVibsFromSources bellow.
            * vibUrls      - A list of URLs to VIB packages
            * metadataUrls - A list of URLs to metadata.zip files
            * vibspecs     - A list of VIB specification strings;
            * fillfrommeta - If metaUrls are passed in with no vibspecs, and
                             fillfrommeta is True, then all the VIBs from
                             the meta/depot will automatically fill the list of
                             VIBs.
            * depotUrls    - The full remote URLs of the depot index.xml
                             or local offline bundle.
            * newestonly   - If newestonly is True, will filter out older VIBs either
                             in depots or in vibspecs matches.
         Returns:
            An instance of VibCollection, which includes all the Vib instances.
         Raises:
            * ValueError - If the same VIB is specified as both a URL and a
                           metadata and their descriptors do not match.
      '''
      vibs = self._getVibsFromUrls(vibUrls)
      if metaUrls or depotUrls:

         if vibspecs:
            vibs += self._getVibsByNameidsFromDepot(metaUrls, vibspecs,
                  depoturls=depotUrls, onevendor=True, newestonly=newestonly)
         elif fillfrommeta:
            vibs += self._getVibsFromDepot(metaUrls, depotUrls,
                  newestonly=newestonly)
      return vibs

   def _setupUpgrade(self, newProfile, force, nosigcheck):
      '''Locate, download, validate esx-update VIB and mount its tardisks to
         kick off profile install/update command with new library.
         Returns path to the ramdisk containing mounted tardisks, or None
         when the VIB cannot be found.

         Side effect: esximage and weasel libraries from the VIB are located
                      and added to sys.path for import.
      '''
      # Import modules used only in this method
      import glob
      import gzip
      from .Utils import HashedStream

      TMP_DIR = '/tmp'
      ESXUPDATE_VIB_NAME = 'esx-update'
      RAMDISK_SIZE = 20
      BUFFER_SIZE = HostImage.HostImage.BUFFER_SIZE

      # Locate esx-update VIB which provide precheck and esximage lib
      updateVib = None
      for _, vib in newProfile.vibs.items():
         if vib.name == ESXUPDATE_VIB_NAME:
            updateVib = vib
            break

      # Going to fallback to normal workflow without valid esx-update VIB
      if not updateVib or not updateVib.remotelocations:
         return None, []

      # There should be only one link to esx-update VIB in depot
      log.info('Downloading esx-update vib from %s' %
               updateVib.remotelocations[0])

      # Use lower pid digits to create a unique ramdisk
      TMP_DIR = '/tmp'
      uniqueName = str(os.getpid())[-7:]
      ramdiskName = '%s-%s' % (ESXUPDATE_VIB_NAME, uniqueName)
      ramdiskPath = os.path.join(TMP_DIR, ramdiskName)

      tardiskNames = []
      try:
         Ramdisk.CreateRamdisk(RAMDISK_SIZE, ramdiskName, ramdiskPath)
         # Download esx-update VIB to temp ramdisk
         localPath = os.path.join(ramdiskPath, updateVib.id)
         try:
            d = Downloader.Downloader(updateVib.remotelocations[0],
                                      local=localPath)
            d.Get()
         except Exception as e:
            raise Errors.VibDownloadError(d.url, None, str(e))
         vibObj = Vib.ArFileVib.FromFile(localPath)

         # Check signature/checksum to make sure the VIB is authentic
         # Signature check is skipped with force or nosigcheck; however,
         # with UEFI secureboot, the check is always executed.
         if HostInfo.IsHostSecureBooted() and (force or nosigcheck):
            log.warn('SecureBoot is enabled, signature of esx-update VIB '
                     'will be checked.')
            checkAcceptance = True
         else:
            checkAcceptance = not (force or nosigcheck)
         if checkAcceptance:
            vibObj.VerifyAcceptanceLevel()

         # Gunzip payloads to the ramdisk
         for payload, sfp in vibObj.IterPayloads():
            checksum = payload.GetPreferredChecksum()
            hashalgo = checksum.checksumtype.replace("-", "")
            sfp = HashedStream.HashedStream(sfp,
                                            checksum.checksum,
                                            hashalgo)
            sfp = gzip.GzipFile(fileobj=sfp, mode='rb')
            # Avoid name clash when mounting
            tarName = '%s-%s' % (payload.name, uniqueName)
            tarPath = os.path.join(ramdiskPath, tarName)

            with open(tarPath, 'wb') as fobj:
               inbytes = sfp.read(BUFFER_SIZE)
               fobj.write(inbytes)
               while inbytes:
                  inbytes = sfp.read(BUFFER_SIZE)
                  fobj.write(inbytes)

            # Mount tardisk and attach to the temp ramdisk
            Ramdisk.MountTardiskInRamdisk(localPath, payload.name, tarPath,
                                          ramdiskName, ramdiskPath)
            tardiskNames.append(tarName)

         # Alter sys.path to be able to import precheck and esximage
         esximagePaths = glob.glob(os.path.join(ramdiskPath, 'lib64', '*',
                                                'site-packages', 'vmware'))
         if not esximagePaths:
            msg = 'Failed to locate esximage library for import in esx-update' \
                  ' VIB'
            raise Exception(msg)
         sys.path.insert(0, esximagePaths[0])
         weaselPath = os.path.join(ramdiskPath, 'usr', 'lib', 'vmware')
         if not os.path.exists(weaselPath):
            msg = 'Failed to locate weasel library for import in esx-update VIB'
            raise Exception(msg)
         sys.path.insert(0, weaselPath)

         log.debug('Added %s and %s to sys.path' %
                   (weaselPath, esximagePaths[0]))
      except Exception as e:
         self._cleanupUpgrade(ramdiskPath, tardiskNames)
         msg = 'Failed to setup upgrade using esx-update VIB: %s' \
               % str(e)
         raise Errors.InstallationError([updateVib.id], msg)

      # Return ramdisk path and tardisk paths for cleanup before exit
      return ramdiskPath, tardiskNames

   def _cleanupUpgrade(self, ramdiskPath, tardiskNames):
      '''Remove esx-update ramdisk and mounted tardisks after profile
         install/update.
      '''
      if ramdiskPath:
         Ramdisk.RemoveRamdisk(os.path.basename(ramdiskPath), ramdiskPath)
      if tardiskNames:
         for tardiskName in tardiskNames:
            Ramdisk.UnmountManualTardisk(tardiskName)

   def UpdateProfileFromDepot(self, metadataUrls, profileName, depotUrls=None,
                              force=False, forcebootbank=False,
                              dryrun=False, checkmaintmode=True,
                              allowDowngrades=False, nosigcheck=False,
                              nohwwarning=False):
      """Installs all the VIBs in an image profile from online or offline zip
         depot sources.
         Hardware precheck of the new image will be executed to ensure
         compatibility, an error occurs with reported issue.
         New esximage library code from the target build will be used.
         As opposed to InstallProfileFromDepot(), VIBs which are
         unrelated to VIBs in the target image profile will be preserved;
         VIBs of higher version in the target image profile will be installed;
         VIBs of loawer version in the target image will be installed only with
         allowDowngrades.
         Parameters:
            Same as InstallProfileFromDepot(), except:
            * allowRemovals   - this parameter is not supported here
            * allowDowngrades - if True, VIBs in the new profile which downgrade
                                existing VIBs will be part of the combined
                                profile.
                                if False (default), VIBs which downgrade will
                                be skipped.
         Returns:
            An instance of InstallResult.
         Raises:
            DowngradeError
            HardwareError
            InstallationError
      """
      Downloader.Downloader.setEsxupdateFirewallRule('true')
      upgradeDir = None
      upgradeTardisks = []
      try:
         depotUrls = depotUrls or []
         newProfile = self.GetProfileFromSources(profileName,
                                                 metadataUrls=metadataUrls,
                                                 depotUrls=depotUrls)
         # Detect unsupported downgrade
         curProfile = self.GetProfile()
         Transaction._checkEsxVersionDowngrade(curProfile, newProfile)

         # Skip precheck and new library download with dryrun or patch the
         # patcher disabled.
         # InstallVibsFromProfile() will get curProfile and newProfile as
         # arguments.
         if dryrun or not PATCH_PATCHER:
            return self.InstallVibsFromProfile(None, None, None,
                                               force, forcebootbank, dryrun,
                                               checkmaintmode, allowDowngrades,
                                               nosigcheck,
                                               curProfile=curProfile,
                                               newProfile=newProfile)

         # Fetch esx-update VIB and setup import
         upgradeDir, upgradeTardisks = self._setupUpgrade(newProfile, force,
                                                          nosigcheck)
         if not upgradeDir:
            log.info('esx-update VIB is not available from the image profile '
                     '%s, calling InstallVibsFromProfile() directly' %
                     profileName)
            return self.InstallVibsFromProfile(None, None, None,
                                               force, forcebootbank, dryrun,
                                               checkmaintmode, allowDowngrades,
                                               nosigcheck,
                                               curProfile=curProfile,
                                               newProfile=newProfile)

         from weasel.util import upgrade_precheck
         from esximage.Transaction import Transaction as NewTransaction

         # Perform hardware precheck
         errorMsg, warningMsg = upgrade_precheck.cliUpgradeAction()
         # Raise exception with errors, --force will not override an error
         if errorMsg:
            msg = 'Hardware precheck of profile %s failed with errors: ' \
                  '%s' % (profileName, errorMsg)
            raise Errors.HardwareError(msg)
         # Warnings can be bypassed with --no-hardware-warning
         if warningMsg:
            if not nohwwarning:
               msg = 'Hardware precheck of profile %s failed with warnings: ' \
                     '%s\n\nApply --no-hardware-warning option to ignore the ' \
                     'warnings and proceed with the transaction.' \
                     % (profileName, warningMsg)
               raise Errors.HardwareError(msg)
            else:
               log.warn('Hardware precheck warnings are ignored with '
                        '--no-hardware-warning')

         # Run InstallProfile() from the new esximage
         log.info('Invoking InstallVibsFromProfile() of the new esximage '
                  'library')
         t = NewTransaction()
         return t.InstallVibsFromProfile(metadataUrls, profileName, depotUrls,
                                         force, forcebootbank, dryrun,
                                         checkmaintmode, allowDowngrades,
                                         nosigcheck)
      finally:
         Downloader.Downloader.setEsxupdateFirewallRule('false')
         self._cleanupUpgrade(upgradeDir, upgradeTardisks)

   def InstallProfileFromDepot(self, metadataUrls, profileName, depotUrls=None,
                               force=False, forcebootbank=False,
                               dryrun=False, checkmaintmode=True,
                               allowRemovals=True, nosigcheck=False,
                               nohwwarning=False):
      '''Installs an image profile from online or offline zip depot sources.
         Hardware precheck of the new image will be executed to ensure
         compatibility, an error occurs with reported issue.
         New esximage library code from the target build will be used.
         During transaction, current image will be completely replaced by the
         new image profile. The VIBs in the image profile will be evaluated
         against the host acceptance level.
            Parameters:
               * metadataUrls   - metadata.zip urls, no op for this function
               * profileName    - Name of image profile to install
               * depotUrls      - The full remote URLs of the depot index.xml
                                  or local offline bundle.
               * force          - Skips most image profile validation checks
               * forcebootbank  - Force a bootbank install even if VIBs are
                                  live installable, no effect for profile
                                  install as it is always reboot required
               * dryrun         - Do not perform stage/install; just report on
                                  what VIBs will be installed/removed/skipped
               * checkmaintmode - Check if maintenance mode is required by live
                                  install, no effect for profile install as
                                  it is bootbank only.
               * allowRemovals  - If True, allows profile installs to remove
                                  installed VIBs.  If False, will raise an error
                                  if profile install leads to removals
               * nosigcheck     - If True, VIB signature will not be validated
               * nohwwarning    - If True, do not show hardware precheck
                                  warnings.
         Returns:
            An instance of InstallResult.
         Raises:
            DowngradeError    - If ESXi version downgrade is not supported
            HardwareError     - If there is a hardware precheck error
            InstallationError - If there is an error in transaction
            NoMatchError      - If profilename matches more than one image
                                profile or if there is no match
            ProfileVibRemoval - If installing the profile causes VIBs to be
                                removed and allowRemovals is False
      '''

      Downloader.Downloader.setEsxupdateFirewallRule('true')
      upgradeDir = None
      upgradeTardisks = []
      try:
         depotUrls = depotUrls or []
         newProfile = self.GetProfileFromSources(profileName,
                                                 metadataUrls=metadataUrls,
                                                 depotUrls=depotUrls)
         # Detect unsupported downgrade
         curProfile = self.GetProfile()
         Transaction._checkEsxVersionDowngrade(curProfile, newProfile)

         # Skip precheck and new library download with dryrun or patch the
         # patcher disabled.
         # InstallProfile() will get curProfile and newProfile as
         # arguments.
         if dryrun or not PATCH_PATCHER:
            return self.InstallProfile(None, None, None,
                                       force, forcebootbank, dryrun,
                                       checkmaintmode, allowRemovals,
                                       nosigcheck, curProfile=curProfile,
                                       newProfile=newProfile)

         # Fetch esx-update VIB and setup import
         upgradeDir, upgradeTardisks = self._setupUpgrade(newProfile, force,
                                                          nosigcheck)
         if not upgradeDir:
            log.info('esx-update VIB is not available from the image profile '
                     '%s, calling InstallProfile() directly' % profileName)
            return self.InstallProfile(None, None, None,
                                       force, forcebootbank, dryrun,
                                       checkmaintmode, allowRemovals,
                                       nosigcheck, curProfile=curProfile,
                                       newProfile=newProfile)

         from weasel.util import upgrade_precheck
         from esximage.Transaction import Transaction as NewTransaction

         # Perform hardware precheck
         errorMsg, warningMsg = upgrade_precheck.cliUpgradeAction()
         # Raise exception with errors, --force will not override an error
         if errorMsg:
            msg = 'Hardware precheck of profile %s failed with errors: ' \
                  '%s' % (profileName, errorMsg)
            raise Errors.HardwareError(msg)
         # Warnings can be bypassed with --no-hardware-warning
         if warningMsg:
            if not nohwwarning:
               msg = 'Hardware precheck of profile %s failed with warnings: ' \
                     '%s\n\nApply --no-hardware-warning option to ignore the ' \
                     'warnings and proceed with the transaction.' \
                     % (profileName, warningMsg)
               raise Errors.HardwareError(msg)
            else:
               log.warn('Hardware precheck warnings are ignored with '
                        '--no-hardware-warning')

         # Run InstallProfile() from the new esximage
         log.info('Invoking InstallProfile() of the new esximage library')
         t = NewTransaction()

         return t.InstallProfile(metadataUrls, profileName, depotUrls,
                                 force, forcebootbank, dryrun,
                                 checkmaintmode, allowRemovals,
                                 nosigcheck)
      finally:
         Downloader.Downloader.setEsxupdateFirewallRule('false')
         self._cleanupUpgrade(upgradeDir, upgradeTardisks)

   def InstallVibsFromSources(self, vibUrls, metaUrls, vibspecs,
                              fillfrommeta=True, keeponlyupdates=False,
                              force=False, forcebootbank=False,
                              stageonly=False, dryrun=False, depotUrls=[],
                              checkmaintmode=True, nosigcheck=False):
      '''Installs VIBs from either direct URLs or from metadata.
         VIBs from both URLs and metadata are merged.  If the same VIB
         is specified as both a URL and a metadata and their descriptors
         do not match, an error results.
            Parameters:
               * vibUrls      - A list of URLs to VIB packages
               * metaUrls     - A list of URLs to metadata.zip files
               * vibspecs     - A list of VIB specification strings; can be
                                <name>, <vendor>:<name>, <name>:<version>,
                                <vendor>:<name>:<version>.  Note that
                                if the spec matches multiple packages,
                                and they are from different vendors,
                                an error results. If they are from the same
                                vendor but different versions, the highest
                                versioned package is taken (since multiple
                                versions cannot be installed at once)
               * fillfrommeta - If metaUrls are passed in with no vibspecs, and
                                fillfrommeta is True, then the newest VIBs from
                                the meta will automatically fill the list of VIBs
                                to install.  If False, no VIBs will be populated
                                from meta when there are no vibspecs.  The VIBs
                                from vibUrls will contribute either way.
               * keeponlyupdates - If True, only VIBs which update VIBs on the
                                   host will be kept as part of the transaction.
                                   The rest will be skipped.
               * force - Skips most image profile validation checks
               * forcebootbank  - Force a bootbank install even if VIBs are
                                  live installable
               * depotUrls      - The full remote URLs of the depot index.xml
                                  or local offline bundle. If URL ends with
                                  '.zip', the URL will be treated as pointing to
                                  an offline bundle and local file path is required.
                                  If the URL does not end in .xml, it will be
                                  assumed to be a directory and 'index.xml' will be
                                  added.
               * dryrun       - Do not perform stage/install; just report on what
                                VIBs will be installed/removed/skipped
               * checkmaintmode - Check maintenance mode if required by live install
               * nosigcheck - If True, VIB signature
                                   will not be validated. A failed validation
                                   raises an exception.
            Returns:
               An instance of InstallResult, with the installed and skipped
               attributes filled out.
            Raises:
               InstallationError - Error installing the VIBs
               NoMatchError      - Cannot find VIBs using vibspecs in metaUrls
               DependencyError   - Installing the VIBs results in invalid image
      '''
      Downloader.Downloader.setEsxupdateFirewallRule('true')

      try:
         curprofile = self.GetProfile()
         vibs = self.GetVibsFromSources(vibUrls, metaUrls, vibspecs, fillfrommeta,
               depotUrls)

         # Scan and keep only the updates
         if keeponlyupdates:
            allvibs = VibCollection.VibCollection()
            allvibs += vibs
            allvibs += curprofile.vibs
            updates = allvibs.Scan().GetUpdatesSet(curprofile.vibs)
            skiplist = list(set(vibs.keys()) - updates)
            log.info("Skipping non-update VIBs %s" % (', '.join(skiplist)))
         else:
            # skip installed VIBs
            skiplist = list(set(vibs.keys()) & set(curprofile.vibs.keys()))
            log.info("Skipping installed VIBs %s" % (', '.join(skiplist)))

         for vibid in skiplist:
            vibs.RemoveVib(vibid)

         log.info("Final list of VIBs being installed: %s" %
                  (', '.join(vibs.keys())))
         inst, removed, exitstate = self._installVibs(curprofile, vibs, force,
                                  forcebootbank, stageonly=stageonly, dryrun=dryrun,
                                  checkmaintmode=checkmaintmode,
                                  nosigcheck=nosigcheck)
      finally:
         Downloader.Downloader.setEsxupdateFirewallRule('false')

      # See #bora/apps/addvob/addvob.c for the vob format string.
      self.host.SendVob("vib.install.successful", len(inst), len(removed))

      return InstallResult(installed=inst, removed=removed, skipped=skiplist,
            exitstate=exitstate)

   def _getVibsFromUrls(self, urls):
      # create VIBs instances from urls
      vibs = VibCollection.VibCollection()
      for url in urls:
         fd, temppath = tempfile.mkstemp(prefix='vib_')
         try:
            d = Downloader.Downloader(url, temppath)
            actual_path = d.Get()
            with open(actual_path, 'rb') as src:
               vib = Vib.ArFileVib.FromFile(src)
               vib.remotelocations.append(url)
               vibs.AddVib(vib)
         except Exception as e:
            raise Errors.VibDownloadError(url, None, str(e))
         finally:
            os.close(fd)
            os.unlink(temppath)
      return vibs

   def _getVibsByNameidsFromDepot(self, metaurls, nameids, depoturls=[],
         onevendor=False, newestonly=False):
      # newestonly - If True, return the latest VIBs matching nameids otherwise
      # all the VIBs for each match are included.
      # create VIB instances from metadata.zip's and
      # a list of <name> / <name>:<version> / <vendor>:<name> VIB spec strings
      #  * onevendor - only allow matches from one vendor for each nameid
      if metaurls:
         self.DownloadMetadatas(metaurls)
      if depoturls:
         self.ParseDepots(depoturls)

      vibs = VibCollection.VibCollection()
      for nameid in nameids:
         try:
            matches = self.vibs.FindVibsByColonSpec(nameid, onevendor=onevendor)
         except ValueError as e:
            raise Errors.NoMatchError(nameid, str(e))

         if len(matches) == 0:
            raise Errors.NoMatchError(nameid, "Unable to find a VIB that matches "
                                      "'%s'." % (nameid))
         if newestonly:
            # Since we have only one vendor and name, if there's multiple matches
            # they must be multiple versions of the same package.
            # Find the highest versioned one for installation.
            vib = max((self.vibs[v].version, self.vibs[v]) for v in matches)[1]
            vibs.AddVib(vib)
         else:
            for v in matches:
               vibs.AddVib(self.vibs[v])

      return vibs

   def _getVibsFromDepot(self, metaurls, depoturls, newestonly=False):
      # Get the all VIBs from metadata and return them
      # If newestonly is True, return the latest vibs in the depots/metas
      if newestonly:
         vibselection = 'newest'
      else:
         vibselection = 'all'
      log.debug("Populating VIB list from %s VIBs in metadata "
                "%s; depots:%s" % (vibselection,
                   ', '.join(metaurls), ', '.join(depoturls)))
      if metaurls:
         self.DownloadMetadatas(metaurls)
      if depoturls:
         self.ParseDepots(depoturls)
      if newestonly:
         result = self.vibs.Scan()
         vibids = result.GetNewestSet()
      else:
         vibids = list(self.vibs.keys())

      return VibCollection.VibCollection((vid, self.vibs[vid]) for vid in vibids)

   def _installVibs(self, curprofile, vibs, force,
                    forcebootbank, stageonly=False,  dryrun=False,
                    checkmaintmode=True, nosigcheck=False, deploydir=None):
      # Add vibs to the current profile, validate it, then install it
      # returns a list of VIB IDs installed, VIB IDs removed and install exitstate
      newprofile = curprofile.Copy()

      try:
         newprofile.AddVibs(vibs, replace=True)
      except KeyError as e:
         msg = 'One or more of the VIBs %s is already on the host.' % \
               (', '.join(v.name for v in vibs.values()))
         raise Errors.HostNotChanged(msg)

      # update profile info
      self._updateProfileInfo(newprofile, 'The following VIBs are installed:\n%s' % (
         '\n'.join('  %s\t%s' % (v.name, v.versionstr) for v in vibs.values())),
                              force)
      exitstate = self._validateAndInstallProfile(newprofile, curprofile, force,
                                   forcebootbank, stageonly=stageonly,
                                   dryrun=dryrun, checkmaintmode=checkmaintmode,
                                   nosigcheck=nosigcheck, deploydir=deploydir)
      adds, removes = newprofile.Diff(curprofile)
      return (adds, removes, exitstate)

   def _validateAndInstallProfile(self, newprofile, curprofile=None,
                                  force=False,
                                  forcebootbank=False,
                                  stageonly=False, dryrun=False,
                                  checkmaintmode=True,
                                  nosigcheck=False,
                                  deploydir=None):
      skipvalidation = force
      systemUpdate = not (stageonly or dryrun)

      if systemUpdate:
         # Prepare general audit information
         if curprofile:
            adds, removes = newprofile.Diff(curprofile)
            auditVibAdds = sorted(adds)
            auditVibRemoves = sorted(removes)
         else:
            # With curprofile missing, removes is empty
            auditVibAdds = sorted(list(newprofile.vibIDs))
            auditVibRemoves = []

         if force or nosigcheck:
            # Start success note, record any attempt to bypass signature check.
            auditStartMsg = self.host.AUDIT_NOTE_NOSIG_IGNORED if \
                            HostInfo.IsHostSecureBooted() else \
                            self.host.AUDIT_NOTE_NOSIG
         else:
            auditStartMsg = None

      # Warn if trying to do an install with no validation
      if force:
         if HostInfo.IsHostSecureBooted():
            msg = ("Secure Boot enabled: Signatures will be checked.")
            log.warn(msg)
            self.host.SendConsoleMsg(msg)
            skipvalidation = False
            nosigcheck = False

         msg = ("Attempting to install an image profile with "
                "validation disabled. This may result in an image "
                "with unsatisfied dependencies, file or package "
                "conflicts, and potential security violations.")
         # See #bora/apps/addvob/addvob.c for the vob format string.
         self.host.SendVob("install.novalidation")
         self.host.SendConsoleMsg(msg)

      if nosigcheck and HostInfo.IsHostSecureBooted():
         msg = ("UEFI Secure Boot enabled: Cannot skip signature checks. Installing "
                "unsigned VIBs will prevent the system from booting. So the vib "
                "signature check will be enforced.")
         log.warn(msg)
         skipvalidation = False
         nosigcheck = False

      # Warn acceptance check is disabled
      checkacceptance = not (skipvalidation or nosigcheck)
      if not checkacceptance:
         msg = ("Attempting to install an image profile bypassing signing and "
                "acceptance level verification. This may pose a large "
                "security risk.")
         self.host.SendConsoleMsg(msg)

      noextrules = force or nosigcheck

      # validate and generate vfatname
      problems = newprofile.Validate(nodeps=force,
                                     noconflicts=force,
                                     allowobsoletes=force,
                                     noacceptance=skipvalidation,
                                     allowfileconflicts=force,
                                     noextrules=noextrules)
      if problems:
         # extract first group of problems with highest priority and raise
         PRIORITY_MAPPING = {0: Errors.ProfileValidationError,
                             1: Errors.VibValidationError,
                             2: Errors.AcceptanceConfigError,
                             3: Errors.DependencyError}

         priority = problems[0].priority
         instances = list(filter(lambda x: x.priority == priority, problems))
         e = PRIORITY_MAPPING[priority](problemset=instances)
         if systemUpdate:
            # Validation failure is a start failure
            self.host.SendAuditEvent(self.host.AUDIT_START_EVENTID,
                                     None,
                                     e,
                                     auditVibAdds,
                                     auditVibRemoves)
         raise e

      newprofile.GenerateVFATNames(curprofile)

      if systemUpdate:
         # Update starts
         self.host.SendAuditEvent(self.host.AUDIT_START_EVENTID,
                                  auditStartMsg,
                                  None,
                                  auditVibAdds,
                                  auditVibRemoves)

      # install and remediate
      try:
         self.host.Stage(newprofile, forcebootbank=forcebootbank,
                         dryrun=dryrun, stageonly=stageonly,
                         checkacceptance=checkacceptance,
                         deploydir=deploydir)
         if systemUpdate:
            self.host.Remediate(checkmaintmode)
         exitstate = Errors.NormalExit()
      except Errors.NormalExit as e:
         exitstate = e
      except Exception as e:
         if systemUpdate:
            # Update ends with error
            self.host.SendAuditEvent(self.host.AUDIT_END_EVENTID,
                                     None,
                                     e,
                                     auditVibAdds,
                                     auditVibRemoves)
         # Not an exit state, raise the error
         raise

      if systemUpdate:
         # Update completed
         self.host.SendAuditEvent(self.host.AUDIT_END_EVENTID,
                                  None,
                                  None,
                                  auditVibAdds,
                                  auditVibRemoves)
      return exitstate

   def RemoveVibs(self, vibids, force=False,
                          forcebootbank=False, dryrun=False,
                          checkmaintmode=True):
      """Removes one or more VIBs from the existing host image.  Validation
         will be performed before the actual stage and install.
            Parameters:
               * vibids         -  list of VIB IDs to remove.
               * force -  Skips most image profile validation checks
               * forcebootbank  -  Force a bootbank install even if VIBs are
                                   live installable
               * dryrun       - Do not perform stage/install; just report on what
                                VIBs will be installed/removed/skipped
               * checkmaintmode - Check maintenance mode if required by live install
            Returns:
               An instance of InstallResult.
            Raises:
               InstallationError - error removing the VIB
               DependencyError   - removing the VIB results in an invalid image
               NoMatchError      - The VIB ID does not exist on the host
      """
      curprofile = self.GetProfile()
      newprofile = curprofile.Copy()
      for vibid in vibids:
         try:
            newprofile.RemoveVib(vibid)
         except KeyError as e:
            raise Errors.NoMatchError('', "VIB ID %s not found in current image profile"
                                      % (vibid))

      # update profile info
      self._updateProfileInfo(newprofile, 'The following VIBs have been removed:\n%s' % (
         '\n'.join('  %s\t%s' % (curprofile.vibs[v].name,
                                curprofile.vibs[v].versionstr) for v in vibids)), force)

      exitstate = self._validateAndInstallProfile(newprofile, curprofile, force,
                                   forcebootbank, dryrun=dryrun,
                                   checkmaintmode=checkmaintmode)
      # See #bora/apps/addvob/addvob.c for the vob format string.
      self.host.SendVob("vib.remove.successful", len(vibids))
      return InstallResult(removed=vibids, exitstate=exitstate)

   def GetProfileFromMetaUrls(self, metadataUrls, profilename, creator=None):
      return self.GetProfileFromSources(profilename, creator, metadataUrls)

   def GetProfileFromSources(self, profilename, creator=None,
                             metadataUrls=[], depotUrls=[]):
      """Returns an image profile from a set of metadata or depots,
         downloading them.  The image profile will have its vibs filled out.
         Parameters:
            * profilename - name of the image profile to look for.
            * creator     - image profile creator.  Used to help refine the
                            search for image profiles if multiple names match.
            * metadataUrls - a list of URLs to metadata.zips
            * depotUrls    - The full remote URLs of the depot index.xml
                             or local offline bundle.
         Returns:
            An ImageProfile instance with vibs VibCollection populated.
         Raises:
            NoMatchError - if profilename is not found or matches more than
                           once.
      """
      self.DownloadMetadatas(metadataUrls)
      if depotUrls:
         self.ParseDepots(depotUrls)

      matches = self.profiles.FindProfiles(name=profilename, creator=creator)
      if len(matches) == 0:
         raise Errors.NoMatchError(profilename,
            "No image profile found with name '%s'" % (profilename))
      elif len(matches) > 1:
         raise Errors.NoMatchError(profilename,
            "More than one image profile found with name '%s'" % (profilename))
      newprofile = list(matches.values())[0]

      # Try to populate VIB instances of image profile
      missing = newprofile.vibIDs - set(self.vibs.keys())
      if missing:
         msg = "VIBs %s from image profile '%s' cannot be found " \
               "in metadata." % (', '.join(list(missing)), newprofile.name)
         raise Errors.NoMatchError(profilename, msg)

      for vid in newprofile.vibIDs:
         newprofile.vibs.AddVib(self.vibs[vid])

      return newprofile

   def InstallProfile(self, metadataUrls, profileName, depotUrls=None,
                      force=False, forcebootbank=False,
                      dryrun=False, checkmaintmode=True,
                      allowRemovals=True, nosigcheck=False,
                      curProfile=None, newProfile=None):
      """Performs installation of an image profile, removing all existing VIBs.
         This is called by InstallProfileFromDepot() with firewall pass-through
         enabled.
         Parameters/Returns/Raise:
            Same as InstallProfileFromDepot(), except:
               curProfile/newProfile - In a dryrun, or when patch the patcher
                                       method is unavailable, these are passed
                                       from InstallProfileFromDepot() so we do
                                       not download metadata and get current
                                       profile for a second time.
      """
      # Only re-download with patch the patcher approach
      if not newProfile:
         newProfile = self.GetProfileFromSources(profileName,
                                                 metadataUrls=metadataUrls,
                                                 depotUrls=depotUrls)

      # Set the image profile acceptance level to the host acceptance value
      # and evaluate the VIBs inside against that
      newProfile.acceptancelevel = self.host.GetHostAcceptance()

      try:
         if not curProfile:
            curProfile = self.GetProfile()
         adds, removes = newProfile.Diff(curProfile)
      except (Errors.AcceptanceGetError, Errors.InstallationError):
         curProfile = None
         adds = list(newProfile.vibIDs)
         removes = list()

      skiplist = list(set(newProfile.vibs.keys()) - set(adds))

      # Detect if any VIBs will be removed due to installing the profile
      if curProfile:
         # Preserve original installdate and signature for skipped VIBs.
         for vibid in skiplist:
            newVib = newProfile.vibs[vibid]
            curVib = curProfile.vibs[vibid]
            newVib.installdate = curVib.installdate
            newVib.SetSignature(curVib.GetSignature())
            newVib.SetOrigDescriptor(curVib.GetOrigDescriptor())

         down, up, gone, common = newProfile.ScanVibs(curProfile.vibs)
         if len(gone) > 0 and not allowRemovals:
            msg = "You attempted to install an image profile which would " \
                  "have resulted in the removal of VIBs %s. If this is " \
                  "not what you intended, you may use the esxcli software " \
                  "profile update command to preserve the VIBs above. " \
                  "If this is what you intended, please use the " \
                  "--ok-to-remove option to explicitly allow the removal." \
                  % gone
            raise Errors.ProfileVibRemoval(newProfile.name, gone, msg)
      else:
         log.warning("No existing image profile, will not be able to detect "
                     "if any VIBs are being removed.")

      exitstate = self._validateAndInstallProfile(newProfile, force=force,
                                   forcebootbank=forcebootbank, dryrun=dryrun,
                                   checkmaintmode=checkmaintmode,
                                   nosigcheck=nosigcheck)
      # See #bora/apps/addvob/addvob.c for the vob format string.
      self.host.SendVob("profile.install.successful", newProfile.name,
            len(adds), len(removes))
      return InstallResult(installed=adds, removed=removes, skipped=skiplist, exitstate=exitstate)

   def InstallVibsFromDeployDir(self, deploydir, dryrun=False):
      """Installs all the VIBs from an image in deployment format,
         such as ISO and PXE.
         VIBs that are not present on the host or upgrade an existing VIB
         will be installed.
         This is currently used during VUM upgrade where all contents
         of an ISO are in a ramdisk.

         Parameters:
            * deploydir - directory with image in deploy format
            * dryrun    - only report on what VIBs will be
                          installed/removed/skipped
         Returns:
            An instance of InstallResult.
         Raises:
            InstallationError
      """
      # Locate image database
      DB_FILE = 'IMGDB.TGZ'
      dbpaths = [os.path.join(deploydir, DB_FILE),
                 os.path.join(deploydir, DB_FILE.lower())]
      db = None
      try:
         for dbpath in dbpaths:
            if os.path.exists(dbpath):
               db = Database.TarDatabase(dbpath)
               db.Load()
               break
      except (Errors.DatabaseIOError, Errors.DatabaseFormatError) as e:
         raise Errors.InstallationError(None, str(e))

      # No image db found
      if not db:
         raise Errors.InstallationError(None, "Failed to locate image database "
                                              "in folder %s." % deploydir)

      newprofile = db.profile
      newprofile.vibs = db.vibs
      curprofile = self.GetProfile()

      # Store source payload localname for staging use
      for vibid, vib in newprofile.vibs.items():
         vibstate = newprofile.vibstates[vibid]
         for payload in vib.payloads:
            if payload.name in vibstate.payloads:
               payload.localname = vibstate.payloads[payload.name]

      # Use new profile name as target profile name
      curprofile.name = newprofile.name

      # Figure out which VIBs to update or add
      updates, downgrades, new, existing = curprofile.ScanVibs(newprofile.vibs)
      skipped = set(existing) | set(downgrades)
      vibstoinstall = VibCollection.VibCollection((vid, newprofile.vibs[vid])
                      for vid in newprofile.vibIDs if vid not in skipped)

      installed, removed, exitstate = self._installVibs(curprofile,
                                                        vibstoinstall,
                                                        False, False,
                                                        dryrun=dryrun,
                                                        deploydir=deploydir)

      return InstallResult(installed=installed, removed=removed,
                           skipped=skipped, exitstate=exitstate)

   def InstallVibsFromProfile(self, metadataUrls, profileName, depotUrls=None,
                              force=False, forcebootbank=False, dryrun=False,
                              checkmaintmode=True, allowDowngrades=False,
                              nosigcheck=False, curProfile=None,
                              newProfile=None):
      """Installs all the VIBs from an image profile that update the host, and
         optionally also downgrade VIBs of same names with allowDowngrades.
         This is called by UpdateProfileFromDepot() with firewall pass-through
         enabled.
         Parameters/Returns/Raises:
            Same as UpdateProfileFromDepot(), except:
               curProfile/newProfile - In a dryrun, or when patch the patcher
                                       method is unavailable, these are passed
                                       from UpdateProfileFromDepot() so we do
                                       not download metadata and get current
                                       profile for a second time.
      """
      # Only re-download with patch the patcher approach
      if not curProfile:
         curProfile = self.GetProfile()
      if not newProfile:
         newProfile = self.GetProfileFromSources(profileName,
                                                 metadataUrls=metadataUrls,
                                                 depotUrls=depotUrls)
      # Use new profile name as target profile name
      curProfile.name = newProfile.name

      # Figure out which VIBs to update or add
      updates, downgrades, new, existing = curProfile.ScanVibs(newProfile.vibs)
      skiplist = set(existing)
      if not allowDowngrades:
         skiplist |= downgrades
      vibstoinstall = VibCollection.VibCollection((vid, newProfile.vibs[vid])
         for vid in newProfile.vibIDs if vid not in skiplist)

      inst, removed, exitstate = self._installVibs(curProfile,
                                                   vibstoinstall, force,
                                                   forcebootbank, dryrun=dryrun,
                                                   checkmaintmode=checkmaintmode,
                                                   nosigcheck=nosigcheck)

      # See #bora/apps/addvob/addvob.c for the vob format string.
      log.debug("Finished self._installVibs")
      self.host.SendVob("profile.update.successful", newProfile.name, len(inst),
                        len(removed))
      log.debug("Finished SendVob")

      return InstallResult(installed=inst, removed=removed, skipped=skiplist,
                           exitstate=exitstate)

   @staticmethod
   def _checkEsxVersionDowngrade(curprofile, newprofile):
      """Check if the profile transaction will mean a major/minor ESXi version
         downgrade. Only downgrade within one release (first two digits are
         same) is supported.
      """
      # Use major and minor number for version compare
      fullcurver = str(curprofile.GetEsxVersion(True).version)
      shortcurver = fullcurver[0:fullcurver.rfind('.')]
      fullnewver = str(newprofile.GetEsxVersion(True).version)
      shortnewver = fullnewver[0:fullnewver.rfind('.')]

      # Block downgrade to older build
      if Version.fromstring(shortcurver) > Version.fromstring(shortnewver):
         msg = "Downgrade ESXi from version %s to %s is not supported." \
                  % (shortcurver, shortnewver)
         raise Errors.DowngradeError(msg)

   @staticmethod
   def _updateProfileInfo(profile, changelog, force=False):
      """Update Image Profile Information. Restore original vendor name
         from description if it was changed to hostname before.
      """
      # If original vendor is in description, restore it.
      splitdstr = profile.description.split('\n', 1)[0].split(':', 1)
      if splitdstr[0] == '(Original Vendor)':
         profile.creator = splitdstr[1]

      if not profile.name.startswith(UPDATEDPROFILESTR):
         profile.name = UPDATEDPROFILESTR + profile.name
      flagstr = ""
      if force:
         flagstr = "WARNING: A --force install has been performed, the image " \
             "may not be valid."
      profile.description = '%s: %s\n%s\n----------\n%s' % \
                            (profile.modifiedtime.isoformat(), changelog,
                             flagstr, profile.description)

