########################################################################
# Copyright (C) 2010 - 2013 VMWare, Inc.
# All Rights Reserved
########################################################################

"""This module contains functions to query host related infomation
"""
import os
import logging

import pyvsilib

log = logging.getLogger('HostInfo')

from vmware.runcommand import runcommand, RunCommandError
from .. import Errors
from .Misc import byteToStr

SMBIOSDUMP_CMD = '/sbin/smbiosDump'
PXE_BOOTING = None
SECURE_BOOT_STATUS_VSI_NODE = "/secureBoot/status"

#
#  Checking for the # of running VMs == 0 is not a substitute for
#  maintenance mode.  It does not cover many use cases such as
#  VMotion in progress.
#
#  Even better would be to enter maintenance mode ourselves -- that has a
#    much stronger guarantee, that no ops are in flight to start/stop a VM,
#    for example.  However, we do not do that for these reasons:
#    i) we'd have to exit maint mode, even for exceptions & errors;
#          - what happens if we have errors exiting maint mode?
#    ii) it's much more testing
#
#    TODO: If we do ever get around to entering maint mode, we should use
#    pyVim/pyVmomi, and connect to hostd via a local ticket (or file a
#    PR to do this).
#
def GetMaintenanceMode():
   """ Returns True if the ESX host is in maintenance mode, and
       False otherwise.
       MaintenanceModeError is thrown if localcli has a non-zero return code,
       such as (most likely) hostd isn't up.
   """

   HOSTD = '/bin/hostd'
   if not os.path.exists(HOSTD):
      # If hostd is not present at all, then we are on a system like ESXCore,
      # where hostd has not been installed. In this - legitimate - case, there
      # is no maintenance mode really, but we still want to install VIBs that
      # require entering in maintenance mode. So our best option is to consider
      # that systems with no hostd are operating in maintenance mode.
      return True

   cmd = ['/bin/localcli', 'system', 'maintenanceMode', 'get']
   rc, out, err = runcommand(cmd, redirectErr=False)
   out = byteToStr(out).strip()
   err = byteToStr(err).strip()
   log.info('localcli system returned status (%d)\nOutput:\n%s\nError:\n%s'
            % (rc, out, err))

   if rc != 0:
      raise Errors.MaintenanceModeError(
         "Unable to determine if the system is in maintenance mode: "
         "localcli returned error (%d).  Please see esxupdate logs for "
         "more details.  To be safe, installation will not continue."
         % rc)

   if out == 'Disabled':
      return False
   elif out == 'Enabled':
      return True

   raise Errors.MaintenanceModeError("Unable to determine if the system "
                                     "is in maintenance mode: localcli "
                                     "returned unexpected result %s.  Please "
                                     "see esxupdate logs for more details.  "
                                     "To be safe, installation will not "
                                     "continue." % out)

def GetBiosVendorModel():
   """ Returns the BIOS vendor name and model strings from pyvsilib.
       returns '', '' if attributes are not available.
   """
   try:
      dmiInfo = pyvsilib.get('/hardware/bios/dmiInfo')
      return dmiInfo.get('vendorName', ''), dmiInfo.get('productName', '')
   except Exception as e:
      log.warn('Failed to get BIOS vendor model: %s' % e)
      return '', ''

def GetBiosOEMStrings():
   """ Return the BIOS OEM String (type 11) entries.
       An empty list is return if none are found.

       @returns: A list of strings

       XXX: As of now the best source for this is the output of smbiosDump.
   """
   label = 'OEM String'
   if os.path.exists(SMBIOSDUMP_CMD):
      rc, out = runcommand([SMBIOSDUMP_CMD])
      out = byteToStr(out)
      if rc != 0:
         log.warn('%s returned nonzero status %d\nOutput:\n%s' % (
            SMBIOSDUMP_CMD, rc, out))
         return []

      heading = None
      indent = 0
      values = list()
      for line in out.split('\n'):
         # we're interested in this specific heading
         if label in line:
            heading = line.lstrip(' ')
            indent = len(line) - len(heading)
         elif not heading:
            continue
         else:
            val = line.lstrip(' ')
            if (len(line) - len(val)) > indent:
               # this line is indented further than the heading
               # that makes it a value
               values.append(val.rstrip())
            else:
               return values
   else:
      log.warn("%s command cannot be found " % SMBIOSDUMP_CMD)
   return []

def IsPxeBooting():
   '''Return True if host is booting from (g)PXE, which is indicated by
      '/sbin/bootOption -rp'. If there was an error in checking bootoption,
      raises InstallationError.
   '''
   global PXE_BOOTING
   #
   # HostSimulator is not considered pxe boot even if the underlying
   # host booted up via pxe.
   #
   if HostOSIsSimulator():
      return 0

   if PXE_BOOTING is None:
      cmd = ['/sbin/bootOption', '-rp']
      rc, outPxe = runcommand(cmd)
      outPxe = byteToStr(outPxe)
      if rc != 0:
         log.warning('[%s] returned nonzero status: %d\nOutput:%s'
                     % (cmd, rc, outPxe))
         msg = 'Unable to get boot type, please see log for detail.'
         raise Errors.InstallationError(None, msg)

      cmd = ['/sbin/bootOption', '-ro']
      rc, outOpts = runcommand(cmd)
      outOpts = byteToStr(outOpts)
      if rc != 0:
         log.warning('[%s] returned nonzero status: %d\nOutput:%s'
                     % (cmd, rc, outOpts))
         msg = 'Unable to get command line options, please see log for detail.'
         raise Errors.InstallationError(None, msg)

      PXE_BOOTING = ('1' in outPxe) or ('statelessCacheBoot' in outOpts)

   return PXE_BOOTING

def HostOSIsSimulator():
   '''Check if the host is running in a simulator.
   '''
   return os.path.exists("/etc/vmware/hostd/mockupEsxHost.txt")

def IsHostSecureBooted():
   '''Check if the host is secure booted
      @return True if secure booted
              False if not secure booted
   '''
   try:
      # For some esximage lib use cases, vsi python module may not
      # be available.
      # In those cases, we want to handle import error
      # elegantly and treat it as security boot not enabled.
      from vmware import vsi

      vsiOut = vsi.get(SECURE_BOOT_STATUS_VSI_NODE)
      return vsiOut['attempted'] != 0
   except ImportError:
      return 0
   except Exception as e:
      log.error(e)
      log.error("Encountered an exception while trying to check secure boot "
                "status. Assuming secure boot is enabled.")
      # We should try to keep our system tight on security, and thus assume that
      # secure boot is enabled here.
      # This assumption shall help us in stopping unauthorized
      # vibs from being installed on the system.
      return 1
