########################################################################
# Copyright (C) 2010 VMWare, Inc.                                      #
# All Rights Reserved                                                  #
########################################################################

''' This Module provides LockFile class based on fcntl.'''

hasfcntl = True
try:
   import fcntl
except ImportError:
   hasfcntl = False

import os
import subprocess
import datetime

class LockFile(object):
   '''File lock based on fcntl.

      The lock is an exclusive lock. A change number is kept in the lock file
      and the caller is responsible to maintain this number.
         Attributes:
            * lockfile - The file path of the lock file.
   '''
   def __init__(self, lockfile):
      self._lockfile = lockfile
      self._lockfobj = None

   def __str__(self):
      return '<Lock: %s>' % (self._lockfile)

   @property
   def lockfile(self):
      return self._lockfile

   def Lock(self, changeno=None, blocking=False):
      """Locks the file lock.  This method blocks until the lock can be
         obtained. If changeno is not None, the changeno will be saved to the
         lock file.
            Parameters:
               * changeno - New change number to write to the lock file
               * blocking - if True, the call will block until the lock is freed up.
                            If False, and the lock cannot be obtained right away,
                            a LockFileError exception will be raised.
            Returns: Change number in the lock file.
            Raises:
               * LockFileError       - If the lock file cannot be opened or
                                       locked.
               * LockFileFormatError - If the content of lock file is
                                       unexpected.
      """
      timestamp = datetime.datetime.now().isoformat()
      if not hasfcntl:
         msg = 'Python module "fcntl" is not available.'
         raise LockFileError(msg)

      try:
         # If we already have an open file descriptor for the serial file,
         # do not open another one. See PR 457302.
         if self._lockfobj:
            fd = self._lockfobj.fileno()
         else:
            fd = os.open(self._lockfile, os.O_RDWR|os.O_CREAT)
            self._lockfobj = os.fdopen(fd, 'r+')
      except Exception as e:
         msg = "Error opening lock file: %s." % e
         raise LockFileError(msg)

      if blocking:
         flag = 0
      else:
         flag = fcntl.LOCK_NB

      try:
         fcntl.lockf(fd, fcntl.LOCK_EX | flag)
      except Exception as e:
         (pid,name) = self._getLockedFileInfo()
         msg = "%s Error locking lock file: %s, it is currently locked by the process"\
               "with pid %s, name: %s" % (timestamp, e, pid, name)
         raise LockFileError(self._lockfile, msg)

      if changeno is not None:
         self.UpdateChangeNo(changeno)
      else:
         try:
            changeno = self._getchangeno()
         except:
            self.Unlock()
            raise

      return changeno

   def Unlock(self):
      """Unlocks the file lock.
            Raises:
               * LockFileError - If the file cannot be unlocked.
      """
      if not hasfcntl:
         msg = 'Python module "fcntl" is not available.'
         raise LockFileError(msg)

      if self._lockfobj is None:
         return

      try:
         fcntl.flock(self._lockfobj, fcntl.LOCK_UN)
         self._lockfobj.close()
         self._lockfobj = None
      except Exception as e:
         msg = "Error unlocking file: %s" % e
         raise LockFileError(self._lockfile, msg)

   def UpdateChangeNo(self, changeno):
      """Update the change number in the lock file. It will try to obtain lock
         first and the lock file content is updated with changeno. Caller need
         to call Unlock to release the lock.
            Parameters:
               * changeno - The new change number in the lock file.
            Raises:
               * LockFileError - If the file lock cannot be acquired or the lock
                                 file content is not updated.
      """
      if self._lockfobj is None:
         fd = os.open(self._lockfile, os.O_RDWR|os.O_CREAT)
         self._lockfobj = os.fdopen(fd, 'r+')

      try:
         fcntl.flock(self._lockfobj, fcntl.LOCK_EX | fcntl.LOCK_NB)
      except IOError as e:
         msg = 'Unable to acquire lock: %s' % (e)
         raise LockFileError(msg)

      try:
         self._lockfobj.truncate(0)
         self._lockfobj.seek(0, 0)
         self._lockfobj.write(str(changeno))
      except Exception as e:
         self.Unlock()
         msg = "Error updating lock change number: %s" % e
         raise LockFileError(msg)

   def _getchangeno(self):
      try:
         self._lockfobj.seek(0, 0)
         # We don't actually expect the serial number to be 512 digits.  That's
         # just there to keep us from reading a whole bunch in the case that
         # this file contains something we don't expect.
         data = self._lockfobj.read(512).strip()
         if not data:
            return 0
         return int(data)
      except ValueError as e:
         msg = "Invalid content in lock file, int content is expected: %s" % (e)
         raise LockFileFormatError(msg)
      except Exception as e:
         msg = "Error reading lock file: %s." % e
         raise LockFileError(msg)

   def _getLockedFileInfo(self):
      '''When the file is acquired by another process, try to get the process's
         pid and name.
      '''
      cmd = 'lsof'
      p = subprocess.Popen(cmd,
                  stdout=subprocess.PIPE, stderr=subprocess.PIPE)
      (out,err) = p.communicate()
      strlines = out.splitlines()
      strlines.reverse()
      filename = os.path.basename(self._lockfile)
      outstr = [s.decode() for s in strlines if filename.encode() in s]
      if outstr:
         return outstr[0].split()[0], outstr[0].split()[1]
      else:
         return 'Unknown', 'Unknown'

class LockFileError(Exception):
   '''Unable to open or lock the lock file.'''

class LockFileFormatError(LockFileError):
   '''The change number in the lock file is not of int value.'''

