########################################################################
# Copyright (C) 2016-2018 VMWare, Inc.                                 #
# All Rights Reserved                                                  #
########################################################################

import logging
import os
import shutil
import tarfile
import tempfile

log = logging.getLogger('Ramdisk')

from vmware.runcommand import runcommand, RunCommandError

from .. import Errors
from .Misc import byteToStr

RAMDISK_ADD_CMD = 'localcli system visorfs ramdisk add -m 0 -M %s ' \
                  '-n %s -p 755 -t %s'
RAMDISK_RM_CMD = 'localcli system visorfs ramdisk list | grep /%s && ' \
                 'localcli system visorfs ramdisk remove -t %s'

def RemoveRamdisk(name, target):
   '''Unmount and remove a ramdisk.
   '''
   try:
      if os.path.exists(target):
         cmd = RAMDISK_RM_CMD % (name, target)
         rc, _ = runcommand(cmd)
         shutil.rmtree(target)
   except RunCommandError as e:
      log.warning('Failed to run %s: %s' % (cmd, e))
   except EnvironmentError as e:
      log.warning('Cannot remove %s directory: %s' % (target, e))

def CreateRamdisk(size, name, target):
   """Create and mount a ramdisk.
   """
   try:
      os.makedirs(target)
      cmd = RAMDISK_ADD_CMD % (size, name, target)
      rc, out = runcommand(cmd)
   except EnvironmentError as e:
      RemoveRamdisk(name, target)
      msg = 'Failed to create ramdisk %s' % name
      raise Errors.InstallationError(None, msg)
   except RunCommandError as e:
      RemoveRamdisk(name, target)
      msg = ('Failed to run %s: %s' % (cmd, e))
      raise Errors.InstallationError(None, msg)
   if rc != 0:
      RemoveRamdisk(name, target)
      msg = ('Failed to run %s: %s' % (cmd, byteToStr(out)))
      raise Errors.InstallationError(None, msg)

def MountTardiskInRamdisk(vibArg, payloadName, tardiskPath, ramdiskName,
                          ramdiskPath):
   """Mount and attach a tardisk to an existing ramdisk.
      Parameters:
         vibArg      - VIB ID or the path to the VIB file; secureMount requires
                       this to verify the tardisk
         payloadName - the name of the payload associated with the tardisk
         tardiskPath - local path of a tardisk
         ramdiskName - name of the ramdisk to attach the tardisk
         ramdiskPath - path to the ramdisk
   """
   SECURE_MOUNT_SCRIPT = '/usr/lib/vmware/secureboot/bin/secureMount.py'

   tardiskMounted = False
   try:
      log.info('Mount tardisk %s in ramdisk %s' % (tardiskPath, ramdiskPath))
      rc, out = runcommand('%s ramdiskMount %s %s %s %s'
                           % (SECURE_MOUNT_SCRIPT, vibArg, payloadName,
                              tardiskPath, ramdiskName))
      if rc == 0:
         tardiskMounted = True
      else:
         log.warn('secureMount returns status %d, output: %s'
                  % (rc, byteToStr(out)))
   except RunCommandError as e:
      log.warn('Failed to execute secureMount: %s' % str(e))

   if not tardiskMounted:
      # Fallback to extraction on failure, this happens during an upgrade
      # where an old secureMount script is present and does not support
      # the ramdiskMount operation.
      log.info('Fallback to extract tardisk %s' % tardiskPath)
      try:
         with tempfile.NamedTemporaryFile() as tmpFd:
            rc, out = runcommand('vmtar -x %s -o %s'
                                 % (tardiskPath, tmpFd.name))
            log.debug('vmtar returns %d, output: %s' % (rc, byteToStr(out)))
            with tarfile.open(tmpFd.name, 'r') as tarFile:
               tarFile.extractall(ramdiskPath)
      except (RunCommandError, tarfile.TarError) as e:
         msg = 'Failed to extract tardisk %s in ramdisk %s: %s' \
               % (tardiskPath, ramdiskPath, str(e))
         raise Errors.InstallationError(None, msg)

def UnmountManualTardisk(tardiskName):
   """Unmount tardisk mounted in tardisks.noauto.
      Such tardisks are mounted to be attached to a ramdisk.
      Parameter:
         tardiskName - filename of the tardisk to be unmounted
   """
   TARDISKS_NOAUTO_PATH = '/tardisks.noauto'

   tardiskPath = os.path.join(TARDISKS_NOAUTO_PATH, tardiskName)
   if os.path.exists(tardiskPath):
      try:
         log.info('Unmounting manual tardisk %s' % tardiskPath)
         os.remove(tardiskPath)
      except Exception as e:
         msg = 'Failed to unmount manual tardisk %s: %s' % (tardiskPath, str(e))
         raise Errors.InstallationError(None, msg)

